function [A,B]=lcd_fitl2reg_fukushima(Zact,Delta,lambda)
% MAR parameter estimate with l2 reguralization for Fukushima model 
%
% * Original Fukushima's model 
%  j(v,t) = a(v)*j(v,t-Delta(v,v)) + sum_u b(v,u)*j(u,t-Delta(v,u)) + epsilon(t)
% 
% [Input] 
% Zact :  current waveforms [Nv Nt]
% Delta : delay matrix (sparse) with unit of time points 
%         (i.e. all the elements must be integer representing dicrete time
%         points)
% lambda : regularization parameters 
%          if set as a scalar, lambda is common to all the vertices
%          if set [], optimal lambda is computed for each vertex  
%          if set as a vector, A and B is computed for all the elements of
%          lambda
% 
% [Output]
%  A  : diagonal parts of MAR matrix (local dynamics)  [Nv,1, length(lambda)]
%  B  : non-digonal parts of MAR matrix (distant interactions) [Nv Nv length(lambda)]
%
% 2016/05/02 O.Yamashita 
% * avoid increasing memory allocation
% 2016/01/28 O.Yamashita 


[Nv,Nt] = size(Zact);
Dmax = max(Delta(:));
dself = Delta(1,1);  % delay of self loop 

if isempty(lambda) | length(lambda) == 1
    B = zeros(Nv,Nv);
    A = zeros(Nv,1);
else
    B = zeros(Nv,Nv,length(lambda));
    A = zeros(Nv,1,length(lambda));
end

for vv = 1 : Nv
    fprintf('process %04d ...\n',vv);
    
    y = [Zact(vv,Dmax+1:Nt)]';
       
    dvec = Delta(vv,:);
    iv = setdiff(find(dvec ~= 0), vv);
    
    X = zeros(Nt-Dmax, length(iv)+1);
    X(:,1) = Zact(vv,Dmax+1-dself:Nt-dself)';
    
    nn = 2;
    for uu = iv
        X(:,nn) = Zact(uu,Dmax+1-dvec(uu):Nt-dvec(uu))';
        nn = nn + 1;
    end
       
    
    if isempty(lambda) | length(lambda) == 1
        [lam,sig,val,a] = wmn_solution(y,X,[],[],struct('flag_plot',0,'lambda',lambda));
        A(vv,:) = a(1);    
        B(vv,iv) = a(2:end);
    else
        [aa] = wmn_solution_fixreg(y, X, [], [], lambda, struct('flag_plot',0));
        A(vv,1,:) = aa(1,:);
        B(vv,iv,:) = aa(2:end,:);
    end
    
end

% 
% figure,
% subplot(2,1,1)
% plot(A);
% subplot(2,1,2)
% plot(B');
% 
% 
% ixeff = find(abs(mean(A,2)) > 0.05) ;
% figure 
% subplot(2,1,1)
% plot(A(ixeff,:)),
% set(gca, 'xtick', [1:length(ixeff)], 'xticklabel', num2str(ixeff));
% subplot(2,1,2)
% imagesc(B(ixeff,ixeff),[-0.05,0.05]);
% set(gca, 'xtick', [1:length(ixeff)], 'xticklabel', num2str(ixeff), 'ytick', [1:length(ixeff)], 'yticklabel', num2str(ixeff));
% cbar






