function h=dmri_plot_dmri_connections_on_brain(dmrifile, brainfile, areafile,  varargin)
%  PLOT DMRI CONNECTIONS ON BRAIN MODEL WITH COARSE REGIONS 
%
% Plotting connections on high-resolution brain model is not visible.
% Thus the area file defining coarse regions such as AAL and brodmann is used. 
% In addition to plot less than 1000 connections is recommended to keep visbility.
% 
% [Input]
%  dmrifile  : connectivity matrix obtained from DMRI processor
%  brainfile : brainfile of VBMEG
%  areafile  : areafile of coarse region parcellation such as AAL and
%              brodmann. AAL is recommended.
%
% 2015/08/26 O.Yamashita
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

parm  = finputcheck(varargin, ...
              { 'Nconnection' 'integer'   []     500; ...
                'lwmax'       'real'      []     5; ...
                'fclr'        'real'      []     [.8 .8 .8];...
                'falpha'      'real'      []     .1;...
                'parcelfile'  'string'    []     ''});
if isstr(parm), error(parm); end
            
Nconnection = parm.Nconnection;   % number of connections to be plotted
lwmax = parm.lwmax;          % maximum linewidth of connections
fclr  = parm.fclr;           % face color of brain
falpha = parm.falpha;        % face alpha of brain
parcelfile = parm.parcelfile;   % required only for fukushima's original format

%---------------------
% brain model
%---------------------

B=load(brainfile);

V = B.V .* 1000; % [m] => [mm]
F0 = B.F.F3;

%------------------------------
%  connectivity
%------------------------------

% IX : parcel indicies to region indices
[C,IX,xyz,Keys] = inner_parcels_to_regions(areafile,dmrifile,parcelfile,V);

% remove empty regions
[IX0,xyz0,Keys0] = inner_remove_null_regions(IX,xyz,Keys);  

% region connectivity
[CC] = inner_get_region_connectivity(C,IX0);
    
% binary region connectivity
[CCbinary] = inner_get_binary_region_connectivity(CC,Nconnection);

% connected_pair
pairs_eff = get_pairs_undir(CCbinary);  



%------------------------------------------
% plot connections on the brain model
%------------------------------------------


% plot patch
h = figure('color',[0 0 0], 'inverthardcopy', 'off');
hp = patch('Faces',F0,'Vertices',V,'FaceColor',fclr,...
          'FaceLighting','phong','EdgeColor','none','facealpha',falpha);
material dull;
hold on;


for vv = 1 : size(pairs_eff,1)

    xyzs = xyz0(pairs_eff(vv,2),:);
    xyzt = xyz0(pairs_eff(vv,1),:);

    uval    = CC(pairs_eff(vv,1),pairs_eff(vv,2));
    dd = xyzt - xyzs;
    dd = abs(dd/norm(dd));
    
   
    % plot lines     
    col = dd;  % RGB color 
    lw  = uval* lwmax;  % linewidth
   
    plot3([xyzs(1) xyzt(1)],[xyzs(2) xyzt(2)], [xyzs(3) xyzt(3)], '-', 'color', col, 'linewidth', lw,...
        'marker', 'o', 'markerfacecolor', 'y', 'markeredgecolor', 'none')
      
end


axis equal; axis tight; axis off;


%------------------------------------------------------
% inner functions 
%

function [C,IX,xyz,Keys] = inner_parcels_to_regions(areafile,dmrifile,parcelfile,V)

if isempty(parcelfile)  % new format
    AC = load(dmrifile);
    C = AC.connectivity_matrix;
    ix = AC.vbmeg_area_ix;
else
    AC = load(dmrifile);
    P = load(parcelfile);
    ix = P.ix_area_mod;
    C = full(AC.Ind);

end

% Corse representaion of connectivity matrix (require area file)
A = load(areafile);
Narea = length(A.Area);

for aa = 1 : Narea
    tmpArea = A.Area{aa};
    xyz(aa,:) = mean(V(tmpArea.Iextract,:),1);  % center of region
    [tmp,jx] = intersect(ix, tmpArea.Iextract);
    IX{aa} = jx;
    Keys{aa} = tmpArea.key;
end

function [IX0,xyz0,Keys0] = inner_remove_null_regions(IX,xyz,Keys)
% erase empty area
nn = 1;
for aa = 1 : size(xyz,1)
    if ~isempty(IX{aa})
        xyz0(nn,:) = xyz(aa,:);
        IX0{nn} = IX{aa};
        Keys0{nn} = Keys{aa};
        nn = nn + 1;
    end
end

function [CC] = inner_get_region_connectivity(C,IX0)

Narea0 = length(IX0);
CC = zeros(Narea0, Narea0);
for aa1 = 1 : Narea0
    for aa2 = 1 : Narea0
        CC(aa1,aa2) = mean(mean(C(IX0{aa1},IX0{aa2})));
    end
end

function [CCbinary] = inner_get_binary_region_connectivity(CC,Nconnection)

tmp = sort(corvec(CC));
thres = tmp(end-Nconnection+1);  % Nconnection-th largest connection
CCbinary = CC >= thres;
