function command = dmri_image_transform_cmd_get(from_file, to_file, to_ref_file, ...
                                                direction, transinfo_dir)
% Get command for transform coordinate system.
%
% [Usage]
%    command = dmri_image_transform_cmd_get(from_file, to_file, to_ref_file, ...
%                                           direction, transinfo_dir);
%
% [Input]
%        from_file : original niftigz file.
%          to_file : transformed and newly create niftigz file.
%      to_ref_file : reference image file of to_file.
%        direction : transform direction   [string]
%                    'fs2struct'          : from freesurfer coord to T1 coord.
%                    'struct2fa'          : from T1 coord to FA coord.
%                    'struct2fa_nointerp' : from T1 coord to FA coord with no interpolation.
%    transinfo_dir : The directory which is including transform files.
%                    This directory is made by dti_transwarp_info_create.m
%
% [Output]
%    command : linux command of transform. give it to system function,
%              transform will be executed.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin ~= 5
    error('Please check input arguments.');
end
if exist(transinfo_dir, 'dir') ~= 7
    error(sprintf('transinfo_dir : %s not found.', transinfo_dir));
end

%
% --- Main Procedure
%
switch(direction)
    case 'fs2struct'
        ref_option = '';
        if ~isempty(to_ref_file)
            if exist(to_ref_file, 'file') == 2
                ref_option = ['-ref ', to_ref_file];
            else
                warning(sprintf('reference file : %s not found.', to_ref_file));
            end
        end

        fs2fa_file = fullfile(transinfo_dir, filesep, 'freesurfer2struct.mat');
        if exist(fs2fa_file, 'file') ~= 2
            error(sprintf('Transform file %s not found.', fs2fa_file));
        end
        
        command = ['flirt -in ' from_file ' ' ref_option, ...
                   ' -out ', to_file, ...
                   ' -applyxfm -init ' fs2fa_file, ' -interp nearestneighbour'];
    case {'struct2fa', 'struct2fa_nointerp'}
        ref_option = '';
        interp_option = '';
        if strcmpi(direction, 'struct2fa_nointerp')
            interp_option = '--interp=nn';
        end

        % reference check
        if ~isempty(to_ref_file)
            if exist(to_ref_file, 'file') == 2
                ref_option = ['-r ', to_ref_file];
            else
                warning(sprintf('reference file : %s not found.', to_ref_file));
            end
        end

        % transform file check
        struct2fa_file = fullfile(transinfo_dir, filesep, 'struct2fa_warp.nii.gz');
        if exist(struct2fa_file, 'file') ~= 2
            error(sprintf('Transform file %s not found.', struct2fa_file));
        end

        % transform command
        command = ['applywarp -i ' from_file, ...
                   ' -o ', to_file, ...
                   ' '   , ref_option, ...
                   ' -w ', struct2fa_file, ...
                   ' '   , interp_option];
end
