function dmri_fs_info_read(freesurfer_dir, fs_info_file)
% create freesurfer information file.
%
% [Usage]
%    dmri_fs_info_read(freesurfer_dir, output_info_file);
%
% [Input]
%    freesurfer_dir : FreeSurfer subject directory.
%      fs_info_file : FreeSurfer information file.
%
%
% * fs_info_file Format
%    FS_wm.lh_wm_file          : 
%         .rh_wm_file          : 
%         .lh_smoothwm_file    : 
%         .rh_smoothwm_file    : 
%         .lh_wm_vertex        : FreeSurfer whitematter verticies(left).
%         .rh_wm_vertex        : FreeSurfer whitematter verticies(right).
%         .lh_smoothwm_vertex  : FreeSurfer smooth whitematter verticies(left).
%         .rh_smoothwm_vertex  : FreeSurfer smooth whitematter verticies(right).
%         .lh_cortex_index     : Cortex index list of verticies(left)
%         .rh_cortex_index     : Cortex index list of verticies(right).
%         .lh_subcortex_index  : SubCortical index list of verticies(left).
%         .rh_subcortex_index  : SubCortical index list of verticies(right).
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin ~= 2
    error('Please check input arguments.');
end

%
% --- Main Procedure
%
disp('Reading FreeSurfer (white/smooth white) matter files...');
if exist(freesurfer_dir, 'dir') ~= 7
    error('FreeSurfer directory not found.');
end

% Create output directory
[p, f, e] = fileparts(fs_info_file);
if exist(p, 'dir') ~= 7
    mkdir(p);
end

% label information created by FreeSurfer
lh_cortex_label_file = [freesurfer_dir, '/label/lh.cortex.label'];
rh_cortex_label_file = [freesurfer_dir, '/label/rh.cortex.label'];


% label list file.(Here, only including cortex label.)
label_list_lh_file = tempname;
label_list_rh_file = tempname;
make_list_file(label_list_lh_file, lh_cortex_label_file);
make_list_file(label_list_rh_file, rh_cortex_label_file);

%
% --- Convert binary to ascii format (whitematter, smooth white matter)
%
lh_white_matter_file = [freesurfer_dir, '/surf/lh.white'];
rh_white_matter_file = [freesurfer_dir, '/surf/rh.white'];
lh_white_matter_asc_file = [p, '/lh_whitematter.asc'];
rh_white_matter_asc_file = [p, '/rh_whitematter.asc'];

lh_smooth_white_matter_file = [freesurfer_dir, '/surf/lh.smoothwm'];
rh_smooth_white_matter_file = [freesurfer_dir, '/surf/rh.smoothwm'];
lh_smooth_white_matter_asc_file = [p, '/lh_smooth_whitematter.asc'];
rh_smooth_white_matter_asc_file = [p, '/rh_smooth_whitematter.asc'];

convert_bin_to_asc(lh_white_matter_file, lh_white_matter_asc_file);
convert_bin_to_asc(rh_white_matter_file, rh_white_matter_asc_file);
convert_bin_to_asc(lh_smooth_white_matter_file, lh_smooth_white_matter_asc_file);
convert_bin_to_asc(rh_smooth_white_matter_file, rh_smooth_white_matter_asc_file);


% Add Cortex label information to White matter surface file(4th)
tmp_lh_labelled_gii_file = [p, '/lh_labelled_whitematter.gii'];
tmp_rh_labelled_gii_file = [p, '/rh_labelled_whitematter.gii'];
lh_labelled_asc_file = [p, '/lh_labelled_whitematter.asc'];
rh_labelled_asc_file = [p, '/rh_labelled_whitematter.asc'];
command3 = ['label2surf --surf=' lh_white_matter_asc_file, ...
            ' --out=' tmp_lh_labelled_gii_file, ...
            ' --labels=' label_list_lh_file];
dmri_system(command3);
command3_1 = ['surf2surf -i ' tmp_lh_labelled_gii_file, ...
              ' -o ' lh_labelled_asc_file, ...
              ' --outputtype=ASCII'];
dmri_system(command3_1);


command4 = ['label2surf --surf=' rh_white_matter_asc_file, ...
            ' --out=' tmp_rh_labelled_gii_file, ...
            ' --labels=' label_list_rh_file];
dmri_system(command4);
command4_1 = ['surf2surf -i ' tmp_rh_labelled_gii_file, ...
              ' -o ' rh_labelled_asc_file, ...
              ' --outputtype=ASCII'];
dmri_system(command4_1);

% Read Labeled white matter surface file
[lh_S, lh_v] = read_asc(lh_labelled_asc_file);
[rh_S, rh_v] = read_asc(rh_labelled_asc_file);

[lh_S2, lh_v2] = read_asc(lh_smooth_white_matter_asc_file);
[rh_S2, rh_v2] = read_asc(rh_smooth_white_matter_asc_file);

% Create FreeSurfer information
FS_wm.lh_wm_asc_file       = lh_white_matter_asc_file;
FS_wm.rh_wm_asc_file       = rh_white_matter_asc_file;
FS_wm.lh_smoothwm_asc_file = lh_smooth_white_matter_asc_file;
FS_wm.rh_smoothwm_asc_file = rh_smooth_white_matter_asc_file;
FS_wm.lh_wm_vertex         = lh_v(:, 1:3);
FS_wm.rh_wm_vertex         = rh_v(:, 1:3);
FS_wm.lh_smoothwm_vertex   = lh_v2(:, 1:3);
FS_wm.rh_smoothwm_vertex   = rh_v2(:, 1:3);
FS_wm.lh_cortex_index      = find(lh_v(:, 4) == 1)';
FS_wm.rh_cortex_index      = find(rh_v(:, 4) == 1)';
FS_wm.lh_subcortex_index   = find(lh_v(:, 4) ~= 1)';
FS_wm.rh_subcortex_index   = find(rh_v(:, 4) ~= 1)';

save(fs_info_file, 'FS_wm');
disp(['Saved:' fs_info_file]);


function make_list_file(label_list_file, label_file)

command = ['echo ' label_file ' >' label_list_file];
system(command);

function convert_bin_to_asc(bin_file, asc_file)

command  = ['mris_convert ', bin_file, ' ', asc_file];
dmri_system(command);
