function dmri_FA_image_create(niftigz_file, ...
                              bval_file, bvec_file, ...
                              brain_mask_file, ...
                              output_dir, ...
                              output_name)
% Create fractional anisotropy image using dtifit command(fsl)
% [Usage]
%    dmri_FA_image_create(niftigz_file,
%                         bval_file, bvec_file,
%                         brain_mask_file,
%                         output_dir,
%                         output_name);
% [Input]
%    niftigz_file    : gzipped Nifti file(.nii.gz)
%                      4D series of data volumes.
%    bval_file       : A text file containing a list of b values applied 
%                      during each volume acquisition. The order of entries 
%                      in this file must match the order of volumes in 
%                      the input data and entries in the gradient 
%                       directions text file. 
%    bvec_file       : A text file containing a list of gradient directions 
%                      applied during diffusion weighted volumes. The order 
%                      of entries in this file must match the order of volumes 
%                      in the input data series. 
%    brain_mask_file : gzipped Nifti file(.nii.gz)
%                      BET binary brain mask.
%                      A single binarised volume in diffusion space 
%                      containing ones inside the brain and zeroes 
%                      outside the brain.
%         output_dir : output directory.
%        output_name : output filename.
%                      output_dir/output_name.nii.gz will be created.
%         
% [See also]
%    http://fsl.fmrib.ox.ac.uk/fsl/fsl4.0/fdt/fdt_dtifit.html
%
% [Output]
%    none
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin ~= 6
    error('Please check input argument.');
end
if exist(niftigz_file, 'file') ~= 2
    error('Specified diffusion weighted data file not found.');
end
if exist(bval_file, 'file') ~= 2
    error('Specified bval_file not found.');
end
if exist(bvec_file, 'file') ~= 2
    error('Specified bvec_file not found.');
end
if exist(brain_mask_file, 'file') ~= 2
    error('Specified brain_mask file(.nii.gz) not found.');
end
if ~ischar(output_dir)
    error('output_dir should be specified by character string.');
end
if ~ischar(output_name)
    error('output_name should be specified by character string.');
end
if exist(output_dir, 'dir') ~= 7
    mkdir(output_dir);
end

%
% --- Main Procedure
%
disp('Creating fractional anisotropy image.');
start = tic;
base_name = [output_dir, '/', output_name];
command = ['dtifit -k ', niftigz_file, ...
           ' -m '       , brain_mask_file, ...
           ' -r '       , bvec_file, ...
           ' -b '       , bval_file, ...
           ' -o '       , base_name];

[status, cmdout] = dmri_system(command, '-echo');
if status ~= 0
    error('Please check input arguments.');
end

% Remove unnecesary files.
delete([base_name, '_L*.nii.gz']);
delete([base_name, '_M*.nii.gz']);
delete([base_name, '_S*.nii.gz']);
delete([base_name, '_V*.nii.gz']);

% Rename
command = ['mv ' base_name, '_FA.nii.gz', ' ', base_name, '.nii.gz'];
[status, comret] = system(command, '-echo');

toc(start);
