function dmri_4D_image_extract(dti_dicom_dir, out_dir, output_basename)
% Extract files from DTI-DICOM directory.
%
% [Usage]
%    dmri_4D_image_extract(dti_dicom_dir, output_basename);
%
% [Input]
%      dti_dicom_dir : DTI dicom directory.
%            out_dir : output directory.
%    output_basename : output file basename.
%
% [Output]
%   none
%
% [Output files]
%  output_dir/output_basename.nii.gz
%     : gzipped nifti diffusion weighted dataset (a 4D image). 
%  output_dir/output_basename.bvec
%     : An ASCII text file containing a list of gradient 
%       directions applied during diffusion weighted volumes. 
%  output_dir/output_basename.bval
%     : An ASCII text file containing a list of b values
%       applied during each volume acquisition.
%
% [Example]
%   dmri_4D_image_extract('/home/cbi/DTI', ...
%                         '/home/cbi/converted_DIT',
%                         'data');
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin < 3
    error('Please check input arguments.');
end
if exist(dti_dicom_dir, 'dir') ~= 7
    error('Specified dicom_dir is not found.');
end
if ~ischar(output_basename)
    error('output_basename is required.');
end

%
% --- Main Procedure
%
disp('Extracting NIfTI file and bval/bvec files from DMRI DICOM files.');
start = tic;
if exist(out_dir, 'dir') ~= 7
   [ret, msg] = mkdir(out_dir);
   if ret ~= 1
       error(msg);
   end
end
out_dir_tmp = tempname(out_dir);
mkdir(out_dir_tmp);

current = pwd;
nifti_file   = [out_dir, '/', output_basename, '.nii'];
bvec_file    = [out_dir, '/', output_basename, '.bvec'];
bval_file    = [out_dir, '/', output_basename, '.bval'];
if exist(bvec_file, 'file') == 2 || ...
   exist(bval_file, 'file') == 2 || ...
   exist([nifti_file, '.gz'], 'file') == 2
%    res = questdlg(...
%          '.nii.gz/bvec/bval already exists. Overwrite?', ...
%          'Confirm', 'Yes', 'No', 'Yes');
%    if strcmpi(res, 'No')
%        disp('processing cancelled.');
%        return;
%    end
end

cd(dti_dicom_dir);
% create nifti file and bvecs/bval files.
[status, cmdout] = system([dcm2nii_cmd, ' -w 1 -o ' out_dir_tmp, ' *'], '-echo');
cd(current);
if status ~= 0
    cd(current);
    error('Failed to convert dicom to nii. Please check VBMEG path.');
end

d = dir(fullfile(out_dir_tmp, '*.nii'));
org_nifti_file = fullfile(out_dir_tmp, d(end).name);
org_bvec_file  = strrep(org_nifti_file, '.nii', '.bvec');
org_bval_file  = strrep(org_nifti_file, '.nii', '.bval');

movefile(org_nifti_file, nifti_file, 'f');
movefile(org_bvec_file,  bvec_file, 'f');
movefile(org_bval_file,  bval_file, 'f');
system(['gzip -f ', nifti_file]);

rmdir(out_dir_tmp, 's');

% back to current path
toc(start);
disp('Finished.');

function prog_path = dcm2nii_cmd
[arch] = computer;
if ispc
    prog_path = which('dcm2niix.exe');
elseif strcmp(arch, 'GLNX86') || strcmp(arch, 'GLNXA64')
    prog_path = which('dcm2niix_linux64.');
else
    prog_path = which('dcm2niix_mac.');
end
    
if ~ispc
    [stat, mess] = fileattrib(prog_path);
    if mess.UserExecute ~= 1
        fprintf('trying to change permission of dcm2niix executable.\n');
        [stat2, mess] = fileattrib(prog_path, '+x');
        if stat2 ~= 1
            error([prog_path ' need to have executable permission.']);
        end
    end
end
