function	[V,xx] = vb_current_vector(V,xx,Lmode)
% Calculate current direction
%   [V,xx] = vb_current_vector(V,xx,Lmode)
% --- INPUT
% V         : Cortex vertex coordinate [NV , 3]
% xx        : Normal direction of cortex [NV , 3]
% V(n,:)    : 3-D coordinate   at n-th vertex
% xx(n,:)   : Normal direction at n-th vertex
% Lmode     : number of independent curent direction
%       = 1 :  current vextor = xx(n,:)
%       = 2 :  current vextor = orthogonal vectors to V(n,:)
%       = 3 :  current vextor = xx(n,:) & orthogonal vectors to xx(n,:)
%
% --- OUTPUT
%  V( NV*Lmode , 3) = repmat(V, [Lmode 1])
% xx( NV*Lmode , 3)
%
% Lmode = 1 :
%   xx = xx(input) : Normal direction
%
% Lmode = 2 : 
%   xx(n,:)      = 1st orthogonal vectors to V(n,:)
%   xx(n + NV,:) = 2nd orthogonal vectors to V(n,:)
%
% Lmode = 3 :
%	xx(n       ,:) =  xx(input) : Normal direction
%	xx(n + NV  ,:) =  1st orthogonal vectors to xx(n,:)
%	xx(n + 2*NV,:) =  2nd orthogonal vectors to xx(n,:)
%               
% if xx == [] &  Lmode = 3 
%	xx(n       ,:) =  [1 0 0]
%	xx(n + NV  ,:) =  [0 1 0]
%	xx(n + 2*NV,:) =  [0 0 1]
%   
% written by M. Sato  2005-8-8
% modified by M. Sato  2006-7-14
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Number of cortex points
NV = size(V,1);

switch	Lmode,
case	1,
	return;
case	2,
	% Transform into spherical coordinate
	[phi,theta,r] = cart2sph(V(:,1),V(:,2),V(:,3));
	
	% Calculate tangential vector to radial direction
	[x1,y1,z1] = sph2cart(phi+0.5*pi, zeros(NV,1),  1); 
	[x2,y2,z2] = sph2cart(  phi,      theta+0.5*pi, 1); 
	xx		   = [ x1,y1,z1 ; x2,y2,z2];

case	3,
	if isempty(xx),
		% Current direction vector
		xx = zeros( 3*NV, 3 ); 
		ix = 1:NV;
		xx(ix       ,1) = 1; % x : [1 0 0]
		xx(ix + NV  ,2) = 1; % y : [0 1 0]
		xx(ix + 2*NV,3) = 1; % z : [0 0 1]
	else
		% Transform into spherical coordinate
		[phi,theta,r] = cart2sph(xx(:,1),xx(:,2),xx(:,3));
		
		% Calculate tangential vector to normal direction xx
		[x1,y1,z1] = sph2cart(phi+0.5*pi, zeros(NV,1),  1); 
		[x2,y2,z2] = sph2cart(  phi,      theta+0.5*pi, 1); 
		xx		   = [ xx; x1,y1,z1 ; x2,y2,z2];
	end
end;

% Replicate Vertex
V  = repmat(V, [Lmode 1]);
