function	[BEM, Vhead, Fhead, Omega, Sout, SPHinfo] = vb_prepare_bem(basis_parm)
% Preparation for leadfield BEM calculation
%
%  [BEM, Vhead, Fhead, Omega, Sout, SPHinfo] = vb_prepare_bem(basis_parm)
%
% --- Output
%  BEM      : Parameters for EEG-BEM 
%  Vhead    : Vertex position of the surface model (NV x 3)
%  Fhead    : Patch data of the surface model (NP x 3)
%  Sout     : Normal of boudary surface (NV x 3
%  Omega    : Solid angle matrix        
%  SPHinfo  : Parameters for Spherical Harmonics
%
% --- 'BEM' should be saved in headfile for EEG-BEM case
%      BEM = []  for MEG cases
% Nsurf         : number of boundary surface = 3 for EEG
% BEM.sigma 	: ( 1 x Nsurf)
%               = conductivity of shell region (ΰƳΨ )
%               = [sigma(1), ..., sigma(Nsurf)]
%                 from innermost to outermost
%
% BEM.R         : ( 1 x Nsurf)
%               = Radius of each surface (for Sphere model EEG case)
%               = [R(1), ..., R(Nsurf)]
% 
% BEM.Nvertex   : ( Nsurf x 2 )
%               = start & end vertex index of each surface
%               = [ start_id(1)     end_id(1)     ; 
%                             ...                 ;
%                   start_id(Nsurf) end_id(Nsurf) ]
%               = [ϥǥå, λǥå]
%
% BEM.Npatch	: ( Nsurf x 2 )
%               = start & end patch index of each surface
%               = [ϥǥå, λǥå]
%
% written by M. Sato  2005-8-8
% modified by T. Sako 2006-8-23
% modified by T. Sako 2006-12-12 - for EEG 3-shell model
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- Boundary surface information
%  Vhead : Vertex position of the surface model (NV x 3)
% XXhead : Normal vector at vertices (NV x 3)
%  Fhead : Patch index of the surface model (NP x 3)

bem_mode = basis_parm.bem_mode;

flg_joined_head = false;
if isfield(basis_parm, 'head_file') && ~iscell(basis_parm.head_file)
  headfile = [basis_parm.proj_root filesep basis_parm.head_file];
else
  headfile = [];
  flg_joined_head = true;
end

global vbmeg_inst;
const = vbmeg_inst.const;
% const.BASIS_MEG_SPHERE;   %  MEG Sphere model (Sarvas)
% const.BASIS_MEG_BEM;      %  MEG BEM
% const.BASIS_MEG_HARMONICS;%  MEG Spherical harmonics expansion
% const.BASIS_EEG_SPHERE;   %  EEG 3-shell Sphere model
% const.BASIS_EEG_BEM;      %  EEG 3-shell BEM

switch bem_mode
case const.BASIS_MEG_SPHERE,
	%  MEG Sphere model (Sarvas)
	BEM   = [];
	Vhead = [];
	Fhead = [];
	Omega = [];
	Sout  = [];

	SPHinfo = [];

case const.BASIS_MEG_BEM, 
	%  MEG BEM 1 shell model
	
	% load head file
	load([headfile]);
	fprintf('# of vertices = %d\n',size(Vhead,1));
	
	if  ~exist('Omega','var') | basis_parm.Recalc == ON,
		  tic;
		  fprintf('--- Solid angle calculation ');
	
		  % ̷׻
		  % (Normal vectors (XXhead) are outwarded here)
		  [Omega, Sout]= vb_solid_angle_grk(Vhead,Fhead,XXhead);
	
		  fprintf('%f[sec]\n',toc);
		  vb_save([headfile],'Omega','Sout');
	end
	
	BEM = [];
	SPHinfo = [];

case const.BASIS_MEG_HARMONICS, 
	%  MEG Spherical harmonics expansion
	
	% load head file
	load([headfile], 'Vhead', 'Fhead', 'XXhead');
	
	fprintf('# of vertices = %d\n',size(Vhead,1));
	
	% load meg file
	meg_file = [basis_parm.proj_root filesep basis_parm.meg_file];
	[pick, Qpick, CoilWeight, Vcenter] = vb_load_sensor(meg_file);
	MEGinfo = vb_load_meg_info(meg_file);
	
	if ~isfield(MEGinfo,'SPHinfo') | basis_parm.Recalc == ON,
	   
		% Center of Head
		if basis_parm.DEBUG == ON,
			% Setting for DEBUG to check Leadfield calculation
			Hcenter = [0 0 0];
		else
			Hcenter = mean(Vhead);
		end

		% average radius from center of head
% 		Rmax = vb_average_radius(Hcenter, Vhead);

		% Change center of coordinate
		Vhead = [Vhead(:,1)-Hcenter(1),  ...
		         Vhead(:,2)-Hcenter(2), Vhead(:,3)-Hcenter(3)];
		pick  = [ pick(:,1)-Hcenter(1),   ...
		          pick(:,2)-Hcenter(2), pick(:,3)-Hcenter(3)];

    % test
		Rmax = vb_average_radius(Hcenter, Vhead);

    % function order
		func_order = basis_parm.func_order;
		
		% Calculate spherical harmonics coefficient
		SPHinfo.harmo_coef = ...
			vb_spherical_harmo_coef(Vhead, XXhead, func_order, Rmax, pick, Qpick);

		% Save SPHinfo to MEGinfo
		SPHinfo.func_order = func_order;
		SPHinfo.radius     = Rmax;
		MEGinfo.SPHinfo    = SPHinfo;
		vb_save(meg_file, 'MEGinfo');
	else
		% Saved result
		SPHinfo = MEGinfo.SPHinfo;
	end
	
	BEM   = [];
	Omega = [];
	Sout  = [];

case const.BASIS_EEG_SPHERE, 
  % EEG 3-shell Sphere model

  % R    : Relative radii of sphere from innermost to outermost
  % sigma: Conductivity  from innermost to outermost
	
	BEM.R     = basis_parm.radius;
	BEM.sigma = basis_parm.sigma;
	
	Vhead = [];
	Fhead = [];
	Omega = [];
	Sout  = [];
	SPHinfo = [];
  
case const.BASIS_EEG_BEM, 
  %  EEG BEM 3-shell model
	
  % check fields
  required_fields = {'head_file', 'sigma'};
  optional_fields = {'LayerTag'};

  for i = 1:length(required_fields)
    if ~isfield(basis_parm, required_fields{i})
      error('incomplete basis_parm : does not exist : %s', required_fields{i});
    end
  end
  
  for i = 1:length(optional_fields)
    if ~isfield(basis_parm, optional_fields{i})
      warning('incomplete basis_parm : does not exist : %s', optional_fields{i});
      basis_parm.LayerTag = char('CSF', 'Skull', 'Scalp');
    end
  end

  % load head file
  new_headfile = vb_util_make_joined_head_file( ...
    basis_parm.proj_root, basis_parm.head_file, basis_parm.sigma, basis_parm.LayerTag);
  
	load([new_headfile]);
	fprintf('# of vertices = %d\n',size(Vhead,1));
  
	if ~exist('BEM','var'), 
    % make BEM from information of headfile
    necessary_info = {'R', 'Sigma', 'Nvertex', 'Npatch'};
    if exist('R', 'var') && exist('Sigma', 'var') && exist('Nvertex', 'var') && exist('Npatch', 'var')
      BEM.R       = R;
      BEM.sigma   = [Sigma 0];  % for vb_bem_matrix
      BEM.Nvertex = Nvertex;
      BEM.Npatch  = Npatch;
    elseif  ~flg_joined_head
      if ~exist('Vhead', 'var') || ~exist('Fhead', 'var')
        error('Head file must have Vhead and Fhead field.(%s)\n', new_headfile);
      end
      
      % single shell head
      if ~exist('R', 'var')             BEM.R = calc_average_radius(Vhead); end;
      if ~exist('Sigma', 'var')     BEM.sigma = [1.0 0];                    end;
      if ~exist('Nvertex', 'var') BEM.Nvertex = [1 length(Vhead)];          end;
      if ~exist('Npatch', 'var')   BEM.Npatch = [1 length(Fhead)];          end;
    else
      % output error information
      for i = 1:length(necessary_info)
        if ~exist(necessary_info{i}, 'var')
          warning('The headfile must have <%s>.\n', necessary_info{i});
        end
      end
      error('Making BEM is incomplete. Check --> %s', headfile);
    end
  end
	
  if  ~exist('Omega','var') | basis_parm.Recalc == ON,
    tic;
    fprintf('--- Solid angle calculation ');
	
    % ̷׻
    % (Normal vectors (XXhead) are outwarded here)
    [Omega, Sout]= vb_solid_angle_grk(Vhead,Fhead,XXhead);
	
    fprintf('%f[sec]\n',toc);
      
    omega_file = vb_util_make_omega_file(new_headfile, Omega, Sout);
  end
  SPHinfo = [];
end

% --- END OF FILE --- %
