function [new_head_file, idx, r_list] = vb_util_make_joined_head_file( ...
  proj_root, headfile_cell, layer_tag, calc_mode, radius)
% joins multiple head-files and makes single head-file which is joined given
% head-files. The new head-file will be named from names of multiple head-files
% with new extention '.omega.mat' <after vb_solid_angle_grk>.
% [usage]
%   [new_head_file, idx, r_list] = vb_util_make_joined_head_file( ...
%     proj_root, headfile_cell, sigma, layer_tag, calc_mode, radius)
% [input]
%       proj_root : <required> directory in which 'new_head_file' will be made
%   headfile_cell : <required> cell array of headfiles.
%                 : if it is not cell, return 'headfile_cell' as 'new_head_file'
%                 : with proj_root
%       layer_tag : <required> names of each layer
%       calc_mode : <optional> calculation method of radius
%                 :  for the function vb_util_join_head_files
%                 :  1 or 2 or 3 or 4
%                 :  [](default depends on vb_util_join_head_files)
%                 :  = 1) mean distance of each head
%                 :  = 2) minimum distance of each head
%                 :  = 3) maximum distance of each head
%                 :  = 4) user-specified radius (radii)
%          radius : <conditionally optional> user-specified radius
%                 :  if calc_mode is 4, this parameter is <required>.
%                 :  []
% [output]
%   new_head_file : joined head-file
%                 :  stored data are as follows:
%                 :        XXhead : [Npoint x 3]
%                 :         Vhead : [NPoint x 3]
%                 :         Fhead : [NPatch x 3] 
%                 :       Nvertex : [NL x 2]  : start and end index
%                 :        Npatch : [NL x 2]  : start and end index
%                 :     layer_tag : {NL} label of each layer
%             idx : index list which is sorted by radius of each layer
%          r_list : radius list which is sorted by their sizes
% [note]
%   @see  vb_util_get_file_body
%   @see  util_join_haed_files
% [history]
%   2006-12-11 (Sako) initial version
%   2007-04-06 (Sako) modified along with the change of vb_util_join_head_files
%   2007-06-14 (Sako) removed some parameter from multi-layer head-mat file
%   2007-11-29 (Sako) added new argument : radius for calc_mode 4
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)
try
  % --- CHECK ARGUMENTS --- %
  if ~exist('proj_root', 'var'), proj_root = []; end
  if ~exist('headfile_cell', 'var'), headfile_cell = []; end
  if ~exist('layer_tag', 'var'), layer_tag = []; end
  if ~exist('calc_mode', 'var'), calc_mode = []; end
  if ~exist('radius', 'var'), radius = []; end
  [ proj_root, headfile_cell, layer_tag, calc_mode, radius] = ...
    inner_check_arguments(proj_root, ...
      headfile_cell, layer_tag, calc_mode, radius);


  % --- MAIN PROCEDURE ------------------------------------------------------- %
  %
  % constants
  OMEGA_EXT = '.omega.mat';
  HEAD_EXT = '.head.mat';
  
  % single file - require no action to join files
  if ~iscell(headfile_cell)
    if ~isempty(strfind(headfile_cell, OMEGA_EXT))
      % do nothing
      idx = [];
      r_list = [];
      new_head_file = [proj_root filesep headfile_cell];
    else  
      new_head_file = ...
        [proj_root filesep vb_util_get_file_body(headfile_cell) HEAD_EXT];
    end
  
    % get r_list idx as return parameters
    if calc_mode == 4
      [r_list] = radius;
    else
      [r_list] = vb_util_calc_head_radius(proj_root, headfile_cell, calc_mode);
    end
    idx = 1:size(r_list,2);
    return;
  end
  
  new_head_file_body = [];
  % make new name of new headfile
  for nf = 1:length(headfile_cell)
    fname_body = vb_util_get_file_body(headfile_cell{nf});
    if nf == 1
      new_head_file_body = fname_body;
    else  
      new_head_file_body = [new_head_file_body '_' fname_body];
    end
  end
  
  new_head_file = [proj_root filesep new_head_file_body HEAD_EXT];
  
%   vb_util_join_head_files( ...
%     headfile_cell, proj_root, new_head_file, sigma, layer_tag );
  [idx, r_list] = vb_util_join_head_files(headfile_cell, proj_root, ...
    new_head_file, layer_tag, calc_mode, radius);
  return;
  
catch
  rethrow(lasterror);
end
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
function [proj_root, headfile_cell, layer_tag, calc_mode, radius] = ...
  inner_check_arguments(proj_root, headfile_cell, layer_tag, calc_mode, radius)

func_ = mfilename;

if isempty(proj_root)
  error('(%s)proj_root is a required parameter', func_);
end

if isempty(headfile_cell)
  error('(%s)headfile_cell is a required parameter', func_);
end

if isempty(layer_tag)
  error('(%s)layer_tag is a required parameter', func_);
end

if isempty(calc_mode)
  % required no action
end

if isempty(radius)
  if calc_mode == 4
    error('(%s)if calc_mode is 4, radius is required parameter', func_);
  end
end
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
