function [HeadInfo] = vb_util_get_specified_layer_headinfo(headfile, layer)
% return HeadInfo (its fields are Vhead, Fhead, XXhead) of specified layer
% [usage]
%   [HeadInfo] = vb_util_get_specified_layer_headinfo(headfile, layer)
% [input]
%   headfile : <required> headfile (.head.mat)
%      layer : <required> number of layer or layer tag that you'd like to get
% [output]
%       HeadInfo : <struct> its fields are ...
%                :  Vhead
%                :  Fhead
%                :  XXhead
% [note]
%   layer is acceptable 'number' and 'string'.
%   if number is given, this function recognizes as number of layer, and
%   if string is given, recognizes as layer tag (e.g. 'Skull')
%   for layer tag, not case-sensitive
% [history]
%   2007-02-14 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('headfile', 'var'), headfile = []; end;
if ~exist('layer', 'var'), layer = []; end;

[layer_number, head_info] = inner_check_arguments(headfile, layer);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%

% --- single layer model
%
if layer_number == 0
  HeadInfo = head_info;
  return;
end

% --- plural layers model
%
IDX_BEGIN = 1;
IDX_END   = 2;
Vhead_begin = head_info.Nvertex(layer_number, IDX_BEGIN);
Vhead_end   = head_info.Nvertex(layer_number, IDX_END);

Fhead_begin = head_info.Npatch(layer_number, IDX_BEGIN);
Fhead_end   = head_info.Npatch(layer_number, IDX_END);

HeadInfo.Vhead  = head_info.Vhead ([Vhead_begin:Vhead_end], :);
HeadInfo.XXhead = head_info.XXhead([Vhead_begin:Vhead_end], :);
HeadInfo.Fhead  = head_info.Fhead ([Fhead_begin:Fhead_end], :);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
function [layer_number, head_info] = inner_check_arguments(headfile, layer)

if isempty(headfile), error('headfile is a required parameter'); end;

if exist(headfile, 'file') ~= 2
  error('cannot read headfile : %s', headfile); end;

if isempty(layer)
  error('neither layer_number nor layer_tag are NOT specified'); end;


% --- load
head_info = load(headfile);

% --- check whether be plural layer model or not
if ~isfield(head_info, 'Nvertex')
  if isfield(head_info, 'Vhead') ...
      && isfield(head_info, 'Fhead') ...
      && isfield(head_info, 'XXhead')
      % single layer model
      layer_number = 0;
    return;
  else
    error('incomplete headfile : %s', headfile);
  end
end

% plural layers
Nlayer = size(head_info.Nvertex, 1);

if isnumeric(layer)
  layer_number = layer;
  if layer_number > Nlayer
    error('specified layer is over : %d', layer_number);
  end
  return;
else
  if isfield(head_info, 'LayerTag')
    specified_tag = upper(layer);
    
    % matching
    for nl = 1:Nlayer
      my_tag = upper(deblank(head_info.LayerTag(nl,:)));
      if strcmp(specified_tag, my_tag)
        layer_number = nl;
        return;
      end
    end      
  else
    error('There is not LayerTag field in this head model : %s', headfile);
  end
end
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %


%%% END OF FILE %%%
