function vb_head_join_files(head_files, file_root, ...
  joined_file, sigma, layer_tag, calc_mode, verbose_swt, check_level)
% join plural single layer head files to single head file serially-concatenated
% [usage]
%   vb_head_join_files(head_files, file_root, 
%     joined_file, sigma, layer_tag, calc_mode, verbose_swt, check_level)
% [input]
%    head_files : <required> cell array of plural "single layer" head files
%               :  {N x 1} or {1 x N}
%
%     file_root : <optional> root directory for 'head_files'
%
%   joined_file : <optional> stored file path which has joined data
%               :  ['(file_root)/joined_head.head.mat']
% 
%         sigma : <optional> electric conductivities of each layer
%               :  [N x 1] or [1 x N]
%               :  [ basis_parm = vb_set_basis_parm; basis_parm.sigma ]
%
%     layer_tag : <optional> names of each layer (cell array)
%               :  {N x 1} or {1 x N}
%               :  [{'CSF', 'Skull', 'Scalp'}]
%
%     calc_mode : <optional> calculation method of radius (R)
%               :  [1] or 2 or 3
%               :  = 1) mean distance of each head
%               :  = 2) minimum distance of each head
%               :  = 3) maximum distance of each head
%
%   verbose_swt : <optional> switch to output verbose message
%               :  [ON] or OFF
% 
%   check_level : <optional> check level
%               :  [0] or 1
%               :  = 0) completely
%               :  = 1) accept its mismatch of a number of files, sigma 
%               :       and layer_tag
%
% [output]
%   void
%
% [note]
%   stored data are as follow:
%      XXhead : [Npoint x 3]
%       Vhead : [NPoint x 3]
%       Fhead : [NPatch x 3] 
%     Nvertex : [NL x 2]  : start and end index
%      Npatch : [NL x 2]  : start and end index
%           R : {NL}  : radii of each layer
%     Rmethod : 'MEAN' or 'MAX' or 'MIN' : calculation method
%       sigma : {NL} electric conductivities of each layer
%   layer_tag : {NL} names of each layer
%
%   @see vb_out_normal.m
%
% [history]
%   2006-12-04 (Sako) initial version
%   2007-04-06 (Sako) added mesh_level
%   2007-04-10 (Sako) changed default value of layer_tag (by cell array)
%   2010-10-25 (Sako) supported the case that XXhead is empty
%   2017-03-27 (rhayashi) specify file_root = [], just use the path of head_files.  
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

try
  % --- CHECK ARGUMENTS --- %
  if ~exist('head_files', 'var'),  head_files = []; end;
  if ~exist('file_root', 'var'),   file_root = []; end;
  if ~exist('joined_file', 'var'), joined_file = []; end;
  if ~exist('sigma', 'var'),       sigma = []; end;
  if ~exist('layer_tag', 'var'),   layer_tag = []; end;
  if ~exist('calc_mode', 'var'),   calc_mode = []; end;
  if ~exist('verbose_swt', 'var'), verbose_swt = []; end;
  if ~exist('check_level', 'var'), check_level = []; end;
  
  [head_files, file_root, joined_file, sigma, layer_tag, calc_mode, ...
    VERBOSE, check_level] = inner_check_arguments( ...
      head_files, file_root, joined_file, sigma, layer_tag, calc_mode, ...
      verbose_swt, check_level);

  % --- head_files' check
  if ~iscell(head_files)
    % do nothing
    warning('%s is not cell so it is not required to join', head_files);
    return;
  end

  % --- MAIN PROCEDURE ------------------------------------------------------- %
  %
  XXhead = [];
  Vhead = [];
  Fhead = [];
  Nvertex = [];
  Npatch = [];
  R = [];

  % Rmethod
  switch calc_mode
    case  1, Rmethod = 'MEAN';
    case  2, Rmethod = 'MIN';
    case  3, Rmethod = 'MAX';
  end
  
  % sort by average radius
  RR = [];
  Nfile = size(head_files,2);
  for nf = 1:Nfile
%     load([file_root filesep head_files{nf}], 'Vhead');
    f = fullfile(file_root, head_files{nf});
    Vhead = vb_head_load_vhead(f);
    RR = [RR vb_util_distance_of_points(Vhead, calc_mode)];
  end
  [R idx] = sort(RR);
  Sigma = sigma(idx);
  LayerTag = layer_tag(:,idx);
  HeadFiles = head_files(:,idx);

  % re-initialize Vhead
  Vhead = [];
  
  begin_v = 1;
  begin_p = 1;
  Npoint  = 0;
  for n = 1:Nfile
    data = load(fullfile(file_root, head_files{n}));
      
%     len_v = size(data.XXhead,1);
%     len_p = size(data.Fhead,1);
    len_v = size(vb_headinfo_get_vhead(data),1);
    len_p = size(vb_headinfo_get_fhead(data),1);

    end_v = begin_v + len_v -1;
    end_p = begin_p + len_p -1;
    Nvertex = [Nvertex; begin_v end_v];
    Npatch  = [Npatch;  begin_p end_p];
    begin_v = end_v + 1;
    begin_p = end_p + 1;
  
%     XXhead = [XXhead; data.XXhead];
%     Vhead = [ Vhead; data.Vhead];
%     Fhead = [ Fhead; data.Fhead + Npoint];
    Vhead = [ Vhead; vb_headinfo_get_vhead(data)];
    Fhead = [ Fhead; vb_headinfo_get_fhead(data) + Npoint];
    
    % support empty XXhead
    if isempty(data.XXhead)
      if ~isempty(data.Vhead) && ~isempty(data.Fhead)
        [new_f, new_v, data.XXhead] = vb_out_normal(data.Fhead, data.Vhead);
      else
        if isempty(data.Vhead), warning('Vhead is empty'); end
        if isempty(data.Fhead), warning('Fhead is empty'); end
        warning('XXhead is empty');
      end
    end
    XXhead = [XXhead; vb_headinfo_get_xxhead(data)];
    
    Npoint = Npoint + len_v;
  end


  vb_save(joined_file, ...
    'XXhead', 'Vhead', 'Fhead', ...
    'Nvertex', 'Npatch', 'R', 'Rmethod', 'Sigma', 'LayerTag');
  
  if VERBOSE
    fprintf('--- saved as current joined file : %s\n', joined_file);
  end

catch
  rethrow(lasterror);
end

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [head_files, file_root, joined_file, sigma, layer_tag, calc_mode, ...
  VERBOSE, check_level] = inner_check_arguments( ...
    head_files, file_root, joined_file, sigma, layer_tag, calc_mode, ...
    verbose_swt, check_level)

% required parameters
if isempty(head_files)
  error('head_files is a required parameter');  
end

% optional parameters
if isempty(file_root)
  file_root = '';
end

if isempty(joined_file)
  joined_file = fullfile(file_root, 'joined_head.head.mat');
end

if isempty(sigma)
    basis_parm = vb_set_basis_parm;
    sigma = basis_parm.sigma;
end

if isempty(layer_tag)
%   layer_tag = char('CSF', 'Skull', 'Scalp');
  layer_tag = {'CSF', 'Skull', 'Scalp'};
end

if isempty(calc_mode)
  calc_mode = 1;  % mean distance
elseif isempty(find(calc_mode == [1 2 3]))
  error('undefined calc_mode : %d', calc_mode);
end
  
if isempty(verbose_swt)
  VERBOSE = OFF;
end

if isempty(check_level)
  check_level = 0;
end

% arrange head files array
head_files = vb_util_arrange_list(head_files, 1);
sigma      = vb_util_arrange_list(sigma, 1);
layer_tag  = vb_util_arrange_list(layer_tag, 1);

% check consistency of data
if check_level == 0
  inner_check_layer_number(head_files, sigma, layer_tag);

  % file check
  inner_check_files(head_files, file_root);
  
elseif check_level == 1
  % do nothing
  
else
  warning('unexpected check_level : %d', check_level);
end
%
% --- end of inner_check_arguments()

% --- inner_check_layer_number()
%
function inner_check_layer_number(head_files, sigma, layer_tag)

Nfiles = size(head_files, 2);
Nsigma = size(sigma, 2);
Nlytag = size(layer_tag, 2);
  
if (Nfiles ~= Nsigma) || (Nsigma ~= Nlytag) || (Nlytag ~= Nfiles) 
  error('not be consistent [Nfiles,Nsigma,Nlytag] = [%d,%d,%d]', ...
    Nfiles, Nsigma, Nlytag);
end
return;
%
% --- end of inner_check_layer_number()

% --- inner_check_files()
%
function inner_check_files(files, root_dir)

Nfiles = size(files, 2);
for fn = 1:Nfiles
  cur_file = fullfile(root_dir, files{fn});
  if exist(cur_file, 'file') ~= 2
    error('cannot read file : %s', cur_file);
  end
end
return;
%
% --- end of inner_check_files()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %
%%% END OF FILE %%%
