function	eeg = vb_eeg_one_shell2(P, R, sigma)
% EEG potential for multiple dipoles at a single EEG sensor
%     in spherical brain model
% Calculate EEG potential for each dipole orientation
%   eeg = vb_eeg_one_shell2(P, R, sigma)
% INPUT
%   Dipole current source
%   P : current dipole position   ( NP x 3 )
%   EEG sensor
%   R : one of sensor position    ( 1 x 3 ) or ( 3 x 1 )
%
% sigma : Conductivity inside the sphere
%
% OUTPUT
%   eeg : EEG Lead field matrix   ( NP x 3 )
%
%   NP : # of dipole points
%
% IEEE TRANSACTIONS ON BIOMEDICAL ENGINEERING, VOL.46, 245-259, 1999
% John C. Mosher, Richard M. Leahy and Paul S. Lewis
%
%  2004-12-15 Made by M. Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Dipole position vector
P1 = P(:,1);
P2 = P(:,2);
P3 = P(:,3);

% Sensor position vector
R  = R(:);
R1 = R(1);
R2 = R(2);
R3 = R(3);

% Difference from dipole to sensor : (NP x 3)
dP = [(R1-P1), (R2-P2), (R3-P3)];

% Square norm    :  (NP x 1)
dd  = sum(dP.^2, 2);
d   = sqrt(dd);
ddd = d.*dd;
pp  = sum(P.^2, 2);

rr  = sum(R.^2);
r   = sqrt(rr);

% Inner product  
dp = sum(dP.*P, 2);
dr = dP*R;
pr = P*R;

% Coefficient terms
f  = dd .* r + d .* dr ;			% Eq.(11)
c1 = 2*dp./ddd + 1./d - 1./r ;  	% Eq.(25)
c2 = 2./ddd + (d + r)./(r.*f) ; 	% Eq.(26)
c3 = ( c1 - c2.*pr )./pp;			% Eq.(27)

% EEG potential
eeg = c3(:,ones(1,3)).*P + c2*R';	% Eq.(27)

eeg = eeg./(4.0*pi*sigma);

return
