function	eeg = vb_eeg_legendre(Xq, Q, Xe, R, sigma, Nmax)
% EEG potential for dipoles 
%     in multilayer spherical forward model 
% Legendre expansion calculation
%  eeg = vb_eeg_legendre(Xq, Q, Xe, R, sigma, Nmax)
% INPUTS (Required):
%   Xq   : dipole location(in meters)                  NP x 3
%   Q    : dipole direction                            NP x 3
%   Xe   : EEG sensors(in meters) on the scalp         M  x 3
%   R    : radii(in meters) of sphere from 
%          innermost to outermost                      NL x 1
%   sigma: conductivity from innermost to outermost    NL x 1
%
% INPUTS (Optional):
%     Nmax : # of terms used in Truncated Legendre Series
%
%  M  = # of sensors
%  NP = # of dipoles
%  NL = # of sphere
%
% OUTPUTS:
%   eeg : EEG Lead field matrix   ( NP x M )
%
% IEEE TRANSACTIONS ON BIOMEDICAL ENGINEERING, VOL.46, 245-259, 1999
% John C. Mosher, Richard M. Leahy and Paul S. Lewis
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

M  = size(Xe,1);	%  # of sensors
NP = size(Xq,1);    %  # of dipoles
NL = length(R);     %  # of sphere

% Radius of dipoles
Rq = sqrt(sum(Xq.^2,2));	% NP x 1

% Radius of outermost layer (Sensor distance from origin)
Rmax = R(NL);  

if R(1)~= min(R)
    fprintf('Head radii must be R(1) < ... < R(NL)\n')
    return
end

if ~all( Rq < R(1) )    
	% check if dipoles within the  innermost layer
    fprintf('All dipoles must be inside innermost layer\n')
    return
end

% EEG potential
eeg = zeros(NP,M);

if nargin < 6, 
    % Default # of Legendre expansion
    Nmax = min( fix(10/log(Rmax/max(Rq))), 200);
end

% Normalized position (unit vector)
Re = sqrt(sum(Xe.^2, 2));
Xe = Xe./Re(:,ones(1,3));		% EEG sensor ( M x 3 )
Xq = Xq./Rq(:,ones(1,3));		% dipole     ( NP x 3 )

% Weight factor of legendre expansion
f = vb_multi_shell_param(R, sigma, Nmax);	% Eq.(16) & (17)

npow = 2*[1:Nmax]+1;

% Coefficient of legendre expansion in Eq.(15)
fn  = ( (npow./[1:Nmax]).*f' )/(4*pi*sigma(NL)*R(NL)^2); % ( 1 x Nmax )

n   = [1:Nmax];
nm1 = n-1;

% loop for dipoles
for i = 1:NP,  
    Xqn = Xq(i,:)';
    Qq  = Q(i,:);
    
    % angle between dipole moment and position vector
    cosalpha = Qq*Xqn;
    
    % angle between dipole position and sensor position vectors
    cosgamma = Xe*Xqn;	% ( M x 1 )
                
    % evaluate legendre polynomial and its derivative
    [Pn, dP] = vb_legendre_grad(Nmax,cosgamma);  % ( Nmax x M )
    
    % power of radius
    ratio = (Rq(i)/Rmax).^nm1;
    
    z = Xe*Qq' - cosgamma*cosalpha;	% ( M x 1 )
    
    w = fn.*ratio;			%  (1 x Nmax)
    
    eeg1 = (w.*n) * Pn;		% (1 x M)
    eeg2 =  w * dP;			% (1 x M)
    
    eeg(i,:) = eeg1.*cosalpha  + eeg2.*z';
end

%eeg = eeg';
