function	headfile = vb_job_head_3shell(varargin)
% Make 1 shell or 3 shell surface model for MEG/EEG
%     CSF (CSF + Gray + White matter) : 1 shell model surface (innermost)
%     Skull (Bone)
%     Scalp (Head Skin) (outermost)
%
%   - CSF is imported from FreeSurfer/Curry surface 
%     or made by using brain model and segmented gray matter file by SPM
%   - Scalp surface is made from face and dilated CSF
%   - Outer Skull surface is made by erosion from Scalp surface
%
% [Usage]
%   headfile = vb_job_head_3shell(parm)
%   headfile = vb_job_head_3shell(proj_root, parm) [old style]
%
% [Input]
%   proj_root : Output file root directory
%   parm      : Parameters explained below
%
% --- Setting for Output file  [relative path from 'proj_root']
% parm.head_file : Base name for head model
%
% --- Setting for Output surface
% parm.Nsurf   : Number of shell (=1 for MEG , =3 for EEG [default])
% parm.Nvertex : number of vertex in each surface 
%              = [2000] (Default)
%  if length(Nvertex) = 1, Nvertex for all surface are set to the same value
%
% --- Setting for MRI input file  (Absolute path)
%   parm.analyze_file : MRI image file
%
% --- Setting for Scalp input file (Absolute path) [There are 2 cases]
%    If the following files are specified at the same time,
%    priority order is fs_scalp_file > face_file
% 1.  parm.face_file     : face file made by vbmeg
% 2.  parm.fs_scalp_file : FreeSurfer Scalp surface ('outer_skin_surface.asc')
%
% --- Setting for CSF input files  [There are 3 cases]
%     (Absolute path except for 'brain_file')
%   We recommend to use both FreeSurfer and Segment files:
%      'freesurf_file' + 'gray_file' + 'brain_file'
%
% 1. FreeSurfer file Case
%   parm.freesurf_file : FreeSurfer CSF surface ('inner_skull_surface.asc')
% 2. Segment file Case
%   parm.gray_file  : Gray matter segment file made by SPM 'Segment'
%   parm.brain_file : Brain model file (.brain.mat)
%        ['brain_file' should be relative path from 'proj_root']
%
%%% --- Advanced Optional Parameter
%   !!! Default parameters are defined by 'vb_set_head_parm'
%
% parm.Scalp_mode 
%     = 0 : Scalp surface is intersection of face and dilated CSF [Default]
%     = 1 : Scalp surface is smoothed face & Skull is the same as mode=0
%     = 2 : Scalp surface is smoothed face & Skull is erosion of face
%
% --- Optional setting for outer Skull surface   (Absolute path)
% parm.fs_skull_file : Outer Skull surface ('/outer_skull_surface.asc')
%
% --- Radius of Morphology operation [mm]
% If the obtained shell model is not good,
% you can adjust the following morphology radius parameters
%
% parm.Radius_scalp = 4 : Thickness of Scalp [mm]
% parm.Radius_skull = 2 : Min Thickness of Skull [mm]
% parm.Radius_fs  : smoothing radius for FreeSurfer surface
%                 = [4 -4 ] Default (No size change)
% parm.Radius_csf : smoothing radius for Gray matter
%                 = [6 6 -7 ] Default (Expand 6 mm)
% parm.Radius     : smoothing radius for CSF surface
%                 = [4 -4 ] Default (No size change)
% 
% parm.Nloop  = 200; Iteration number for fit boundary
% parm.Nlast_csf   = 10 ; Iteration number for CSF smoothing by spring model
%                    As Nlast becomes larger, surface becomes smoother
%                    but it shrinks inward
% parm.Nlast_scalp  = 0 ; Iteration number for Scalp
% parm.Nlast_skull  = 0 ; Iteration number for Skull
% (Gray matter parameters)
% parm.Glevel      = [0.8];  level of gray matter selection threshold
% parm.Radius_gray = [-2 2]; Radius to remove noise in gray matter image
% (Brain model parameters)
% parm.Radius_fill = [6];    Cortex fill radius
%
% Masa-aki Sato 2009-10-7
% Masa-aki Sato 2010-09-15  add FreeSurfer case
% Masa-aki Sato 2010-10-27  modified procedures
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if nargin == 1
    proj_root = [];
    parm = varargin{1};
elseif nargin == 2
    proj_root = varargin{1};
    parm = varargin{2};
end

proj_root = vb_rm_trailing_slash(proj_root);

% Default Parameters
head_parm = vb_set_head_parm;

% Number of surface
if ~isfield(parm,'Nsurf'), 
	Nsurf = head_parm.Nsurf;
else
	Nsurf = parm.Nsurf; 
end;

% Number of vertex in one surface [2000]
if ~isfield(parm,'Nvertex'), 
	switch	Nsurf
	case	1
		Nvertex = [6000]; 
	case	3
		Nvertex = head_parm.Nvertex; 
	end
else
	Nvertex = parm.Nvertex;
end;
if length(Nvertex)==1, Nvertex = repmat(Nvertex,[1,Nsurf]); end;

% Set Default Radius parameters
if ~isfield(parm,'Radius_fs'),   parm.Radius_fs    = head_parm.Radius_fs ;end;
if ~isfield(parm,'Radius_csf'),  parm.Radius_csf   = head_parm.Radius_csf;end;
if ~isfield(parm,'Radius_scalp'),parm.Radius_scalp = head_parm.Radius_scalp;end
if ~isfield(parm,'Radius_skull'),parm.Radius_skull = head_parm.Radius_skull;end
if ~isfield(parm,'Skull_ratio'), parm.Skull_ratio  = head_parm.Skull_ratio ;end

if ~isfield(parm,'Radius_final'),parm.Radius_final = head_parm.Radius_final;end
if ~isfield(parm,'Radius'),      parm.Radius       = head_parm.Radius     ;end;
if ~isfield(parm,'Radius_fill'), parm.Radius_fill  = head_parm.Radius_fill;end;
if ~isfield(parm,'Radius_gray'), parm.Radius_gray  = head_parm.Radius_gray;end;
if ~isfield(parm,'Glevel'),      parm.Glevel       = head_parm.Glevel     ;end;

if ~isfield(parm,'Nloop'),       parm.Nloop       = head_parm.Nloop      ;end;
if ~isfield(parm,'Nlast_csf'),   parm.Nlast_csf   = head_parm.Nlast_csf  ;end;
if ~isfield(parm,'Nlast_scalp'), parm.Nlast_scalp = head_parm.Nlast_scalp;end;
if ~isfield(parm,'Nlast_skull'), parm.Nlast_skull = head_parm.Nlast_skull;end;

if ~isfield(parm,'Scalp_mode'), parm.Scalp_mode = head_parm.Scalp_mode;end;
if ~isfield(parm,'vstep'), 		parm.vstep = head_parm.vstep; end;

step = parm.vstep;

% Plot mode
if ~isfield(parm,'plot_mode'), 
	plot_mode = 0;
else
	plot_mode = parm.plot_mode; 
end;

% Brain & image file name
analyzefile = parm.analyze_file;
if isfield(parm, 'brain_file')
    brain_file = fullfile(proj_root, parm.brain_file);
    brain_parm = parm;
    brain_parm.brain_file = brain_file;
else
    brain_file = [];
end

%
% --- Definition of output files
%

[headfile, basename] = change_name(parm.head_file);

% Output headfile name
headfile  = fullfile(proj_root, headfile);

% CSF surface(Nsurf=1,3 common)
headsave{1} = fullfile(proj_root, change_name([basename '_CSF'], Nvertex(1)));

% Skull, Scalp (for 3shell model)
if Nsurf == 3
headsave{2} = fullfile(proj_root, change_name([basename '_Skull'], Nvertex(2)));
headsave{3} = fullfile(proj_root, change_name([basename '_Scalp'], Nvertex(3)));
end

% Read MRI image size
[Vdim, Vsize] = analyze_hdr_read(analyzefile);

[Vface,Fface,Vs,Fs] = load_face(parm, Vdim, Vsize);

%%% DEBUG %%%
if plot_mode > 0
	debug_plot(Vface, Fface, analyzefile);
end

%
% --- Make CSF surface
%
parm.Nvertex = Nvertex(1);
parm.Nlast = parm.Nlast_csf;
% Add margin to image for Morphology
DW  = 20;
%	Vhead = Vhead + step*DW;
%	Vhead = Vhead - step*DW;

if isfield(parm,'freesurf_file') && ~isempty(parm.freesurf_file)
%
% --- Import FreeSurfer surface file
    disp(['--- Loading FS-CSF model file... ']);
    [Vhead,Fhead] = vb_fs_load_surface(parm.freesurf_file);
    [Fhead,Vhead,XXhead] = vb_out_normal_surf(Fhead,Vhead);
    Vhead = Vhead * 0.001;
		
	%%% DEBUG %%%
	if plot_mode > 0
		debug_plot(Vhead, Fhead, analyzefile);
	end
	
	if ~isempty(brain_file)
		% Make mask from brain
		B = vb_make_head_surf_model(brain_parm);
		Dim  = size(B);
		
		Bcsf = zeros(Dim + 2*DW);
		Bcsf((1:Dim(1))+DW, (1:Dim(2))+DW, (1:Dim(3))+DW) = B;
		Dim = Dim + 2*DW;
	else
		Dim = ceil((Vdim .* Vsize)/step) + 2*DW;
		%Dim   = vb_mask_image_size(Vdim,Vsize,step);
	end
	
	% Make mask from FreeSurfer head surface
	Vhead = vb_spm_right_to_analyze_right_mm(Vhead, Vdim, Vsize);
	Vhead = Vhead + step*DW;
	
	% Make mask from FreeSurfer surface
	B = vb_surf_to_filled_mask(Vhead, Fhead, Dim, step);
	
	% Smooth & Shrink FreeSurfer surface
	B = vb_morphology_operation(B, parm.Radius_fs, step);

	% Merge cortex expand surface & FreeSurfer surface mask
	if ~isempty(brain_file)
		% Make mask from brain
		%Bcsf = vb_make_head_surf_model(brain_parm);
		
		if vb_matlab_version > 6,
			B = int8(B) + int8(Bcsf);
		else
			B = double(B) + double(Bcsf);
		end
	end
	
	% Make mask image from face surface and shrinks it by (Scalp+Skull) radius
	if ~isempty(Vs)
		Vs     = Vs + step*DW;
		Bface  = vb_face_to_mask(Vs, Fs, Dim, step);
		Radius = [- parm.Radius_scalp, - parm.Radius_skull];
		Bface  = vb_morphology_operation(Bface, Radius , step);
		
		% Intersection of CSF and shrinked face
		if vb_matlab_version > 6,
			B = int8(B) .* int8(Bface);
		else
			B = double(B) .* double(Bface);
		end
	end
	
	% Smooth surface with vertex number 'parm.Nvertex'
	%   ( Morphology by parm.Radius )
	[Vhead, Fhead, XXhead, B] = vb_mask_to_surf_expand(B, parm);
	Vhead = Vhead - step*DW;
	
	Vhead = vb_analyze_right_mm_to_spm_right(Vhead, Vdim, Vsize);
	
% 	disp(['--- Save CSF model file: ']);
% 	disp([headsave{1}]);
	vb_fsave(headsave{1},'Vhead','Fhead','XXhead');
	
	if Nsurf==1
		vb_plot_slice_head(headsave{1}, analyzefile, brain_file);
	end
elseif	isfield(parm,'gray_file') && ~isempty(parm.gray_file)
%	
% ---  Make mask from brain surface and gray image
	[B] = vb_make_head_surf_model(brain_parm);
	
	%Dim   = vb_mask_image_size(Vdim,Vsize,step);
	Dim   = size(B);
	
	% Make mask image from face surface and shrinks it by (Scalp+Skull) radius
	if ~isempty(Vs)
		Bface = vb_face_to_mask(Vs, Fs, Dim, step);
		Radius = [- parm.Radius_scalp, - parm.Radius_skull];
		Bface = vb_morphology_operation(Bface, Radius , step);
		
		% Intersection of CSF and shrinked face
		if vb_matlab_version > 6,
			B = int8(B) .* int8(Bface);
		else
			B = double(B) .* double(Bface);
		end
	end

	% Smooth surface with vertex number 'parm.Nvertex'
	% Morphology by parm.Radius
	[Vhead, Fhead, XXhead, B] = vb_mask_to_surf_expand(B, parm);
	Vhead = vb_analyze_right_mm_to_spm_right(Vhead, Vdim, Vsize);

	if isempty(XXhead), return; end;
	
% 	disp(['--- Save CSF model file: ']);
% 	disp([headsave{1}]);
	vb_fsave(headsave{1},'Vhead','Fhead','XXhead');
	
	if Nsurf==1,
		vb_plot_slice_head(headsave{1}, analyzefile, brain_file);
	end
else
	error('No CSF input file');
end


% 1shell case
if Nsurf==1,
    movefile(headsave{1}, headfile);

    % save parameter
    head_parm = parm;
    vb_save(headfile, 'head_parm');

    vb_disp('--- Save Head model file: ');
    vb_disp(headfile);
	project_file_saving(parm);
	return; 
end;

if isempty(Vface), error('Scalp file is needed for 3 shell model'); end

% Load CSF surface
load(headsave{1},'Vhead','Fhead')

%%% DEBUG %%%
if plot_mode > 0
	debug_plot(Vhead, Fhead, analyzefile);
	if plot_mode == 1, return; end;
end

%
% --- Make Scalp : intersection of face surface and dilated CSF surface
%
[Rmax,Rmin] = vb_distance_brain_face(Vhead,Vface);

fprintf('Max distance from face to CSF = %5.1f\n',Rmax)
fprintf('Min distance from face to CSF = %5.1f\n',Rmin)

Rstep = round(Rmax/2);
Radius = [Rstep  ceil(Rmax + 2 - Rstep)];
fprintf('Expansion Radius from CSF to Scalp = %3.0f\n',sum(Radius))

% --- Make intersection of dilated brain and face surface

[B ,Bcsf, Bface] = ...
	vb_intersect_face_csf(Vhead,Fhead,Vface,Fface,Radius,step,Vdim,Vsize,DW);

%%% DEBUG %%%
if plot_mode > 1
	zindx = [40:20:200];
	zindx = ceil(zindx/step);
	vb_plot_slice( B, [], zindx, 1);
	vb_plot_slice( Bcsf, [], zindx, 1);
	vb_plot_slice( Bface, [], zindx, 1);
	if plot_mode == 2, return; end;
end

parm.Nlast = parm.Nlast_scalp;

if parm.Scalp_mode == 0
	% --- Extract smooth surface from intersect_face image
	parm.Nvertex = Nvertex(3);
	parm.Radius  = parm.Radius_final;
	
	[Vhead, Fhead, XXhead, B] = vb_mask_to_surf_expand(B, parm);
	
	Vhead = Vhead - step*DW;
	Vhead = vb_analyze_right_mm_to_spm_right(Vhead,Vdim,Vsize);
	
% 	disp(['--- Save Scalp model file: ']);
% 	disp([headsave{3}]);
	vb_fsave(headsave{3},'Vhead','Fhead','XXhead');
else
	% --- Extract closed face surface
	parm.Radius  = parm.Radius_final;
	parm.Nvertex = Nvertex(3);
	
	[Vhead, Fhead, XXhead, Bface] = vb_mask_to_surf_expand(Bface, parm);
	
	Vhead = Vhead - step*DW;
	Vhead = vb_analyze_right_mm_to_spm_right(Vhead,Vdim,Vsize);
	
% 	disp(['--- Save Face model file: ']);
% 	disp([headsave{3}]);
	vb_fsave(headsave{3},'Vhead','Fhead','XXhead');
	
	switch	parm.Scalp_mode 
	case	1
		% --- smooth Scalp surface
		parm.Radius  = parm.Radius_final;
		B = vb_morphology_operation(B, parm.Radius, step);
	case	2
		% --- smooth Face surface
		B = Bface;
	end
end

clear Bface

%%% DEBUG %%%
if plot_mode > 2
	debug_plot(Vhead, Fhead, analyzefile);
	if plot_mode == 3, return; end;
end
%
% --- Make Outer Skull surface : erosion from Scalp surface
%

if isfield(parm,'fs_skull_file') && ~isempty(parm.fs_skull_file)
%
% --- intersection of Skull surface and dilated CSF surface
%
	% Load Skull & CSF surface
	load(headsave{1},'Vhead','Fhead')
	[V,F] = vb_fs_load_surface(parm.fs_skull_file);
	V = V * 0.001;
	
	% CSF Expansion Radius
	Radius(2) = Radius(2) - parm.Radius_scalp;
	
	% --- Make intersection of dilated CSF and Skull surface mask
	[B ,Btmp, Bskl] = ...
		vb_intersect_face_csf(Vhead,Fhead,V,F,Radius,step,Vdim,Vsize,DW);
	
	clear Btmp
	switch	parm.Scalp_mode 
	case	{0,1}
		% --- smooth Skull surface
		parm.Radius  = parm.Radius_final;
		B = vb_morphology_operation(B, parm.Radius, step);
	case	2
		% --- smooth Face surface
		B = Bskl;
	end
else	
	% --- Erosion from Scalp surface
	fprintf('Max width of Scalp = %5.1f\n',parm.Radius_scalp)
	
	B = vb_morphology_operation(B, - parm.Radius_scalp , step);
	%fprintf('# of Scalp mask %d\n',length(find(B(:)>0)))
end
% --- Set min dilation radius from CSF
Rskull = Rmin * parm.Skull_ratio;
fprintf('Min width of Skull = %5.1f\n',Rskull)

Bcsf = vb_morphology_operation(Bcsf, Rskull, step);
%fprintf('# of CSF   mask %d\n',length(find(Bcsf(:)>0)))

% Make union of two images
if vb_matlab_version > 6,
	B  = int8(B) + int8(Bcsf);
	ix = find( B(:) > 0 );
	B  = zeros(size(B),'int8');
	B(ix) = 1;
else
	B  = double(B) + double(Bcsf);
	ix = find( B(:) > 0 );
	B  = zeros(size(B));
	B(ix) = 1;
end

%fprintf('# of Skull   mask %d\n',length(ix))

% --- Extract smooth Skull surface from mask image [analyze_right_mm]
parm.Radius  = parm.Radius_final;
parm.Nvertex = Nvertex(2);
parm.Nlast   = parm.Nlast_skull;

[Vhead, Fhead, XXhead, B] = vb_mask_to_surf_expand(B, parm);

Vhead = Vhead - step*DW;
Vhead = vb_analyze_right_mm_to_spm_right(Vhead,Vdim,Vsize);

% disp(['--- Save Skull model file: ']);
% disp([headsave{2}]);
vb_fsave(headsave{2},'Vhead','Fhead','XXhead');

% join 3 surface into one file
vb_head_join_files(headsave, [], headfile);

%
% --- Check 3 surface on MRI
%

vb_plot_slice_head(headsave, analyzefile, brain_file);

%return

% --- Check Scalp surface with face
load(headsave{3},'Vhead','Fhead');

if exist('surf_face','var') && isfield(surf_face,'V_reduce')
	Vface = surf_face.V_reduce;
	Fface = surf_face.F_reduce;
end

figure
subplot(1,2,1)
vb_plot_surf(Vface,Fface,[],[],1,1);
hold on
vb_plot_surf(Vhead,Fhead,'none','k');
alpha 0.8;

title('Scalp surface on face')

view([ -210 30]);

subplot(1,2,2)
vb_plot_surf(Vhead,Fhead,[],[],1,1);

title('Scalp surface')

view([ -210 30]);

% save parameter
head_parm = parm;
vb_save(headfile, 'head_parm');
vb_disp('--- Save Head model file: ');
vb_disp(headfile);

% save project
project_file_saving(parm);

% remove temporary files
for k=1:length(headsave)
    delete(headsave{k});
end

return

function	project_file_saving(parm)
%
% --- Save project file
%
proj_file = get_project_filename;
if isempty(proj_file)
    return;
end
project_file_mgr('load', proj_file);
project_file_mgr('add', 'head_parm', parm);


return

function	[fnew, basename] = change_name(fold, N)
%
% --- change_name
%

fext = '.head.mat';
ix_ext = findstr(fold, fext);

if ~isempty(ix_ext),
	basename = fold(1:ix_ext-1);
else
	basename = fold;
end

if nargin == 2
	fnew = sprintf('%s_%d%s', basename, N, fext);
else
	fnew = sprintf('%s%s', basename, fext);
end

return

function	[V,F,Vs,Fs] = load_face(parm, Vdim, Vsize)

% Load Face surface
if isfield(parm,'fs_scalp_file') && ~isempty(parm.fs_scalp_file)
	[V,F] = vb_fs_load_surface(parm.fs_scalp_file);
	V = V * 0.001; % [mm] -> [m]
elseif isfield(parm,'face_file') && ~isempty(parm.face_file)
	load(parm.face_file,'surf_face');
	V = surf_face.V;
	F = surf_face.F;
else
	V  = []; F  = []; 
	Vs = []; Fs = []; 
	return
end

% Make surface closed
[F,V] = vb_close_surf(F,V);

% Coordinate transform to analyze_right_mm
Vs = vb_spm_right_to_analyze_right_mm(V, Vdim, Vsize);
Fs = F;

% Min z-coordinate for face mask [mm]
zmin = 2; 
Vs(:,3) = max(Vs(:,3),zmin);

return

function	debug_plot(V, F, analyzefile)
	[B, Vdim, Vsize] = vb_load_analyze_to_right(analyzefile);
    Vana = vb_spm_right_to_analyze_right( V, Vdim, Vsize);
	
	zindx = 0.1*[2:2:8];
	xindx = 0.1*[2:2:8];
	
    Vmin = min(Vana);
    Vmax = max(Vana);
	zindx = ceil( zindx*(Vmax(3) - Vmin(3)) + Vmin(3));
	xindx = ceil( xindx*(Vmax(1) - Vmin(1)) + Vmin(1));
	zindx = zindx( zindx > 0 & zindx <= Vdim(3) );
	xindx = xindx( xindx > 0 & xindx <= Vdim(1) );
	
    vb_plot_slice_surf(B, Vana, F, zindx, 'z', [2 2]);
    vb_plot_slice_surf(B, Vana, F, xindx, 'x', [2 2]);

return

ix = find( Vs(:,3) >= zmin);
Fs = vb_patch_select2(ix,F,size(Vs,1));
