function [Zact, Jinfo, Jext] = vb_job_current(varargin)
% Estimate current time course by using inverse filter. 
% (VBMEG public function)
%
% [syntax]
% [Zact,Jinfo,Jext] = vb_job_current(curr_parm)
% [Zact,Jinfo,Jext] = vb_job_current(proj_root,curr_parm) [old style]
%
% [input]
% proj_root: <<string>> VBMEG project root directory. 
% curr_parm: <<struct>> Current estimation parameters. 
% --- fields of curr_parm
%  bayesfile    : <<string>> Model parameter filename (.bayes.mat). This
%                 file is required for current estimation and obtained by
%                 vb_job_vb before running this function. 
%  currfile     : <<string>> Cortical current filename (.curr.mat),
%                 specifying the file to which current estimation result
%                 is saved. 
%  area_key     : <optional> <<string>> Area ID on which cortical current
%                 is estimated. This parameter will be useful if you have
%                 some knowledge about region of interested. 
%  areafile     : <optional> <<string>> Rquired if cortical area is
%                 specified by field 'area_key'. 
%  trial_average: <optional> <<bool>> If true (default), cortical current
%                 averaged across all trials. If false, cortical current
%                 is estimated for all trials. 
%  jactdir      : <optional> <<string>> Directory for saving trial
%                 current files. This is the relative path from the
%                 cortical current file. If trial_average == false &
%                 jactdir is specified, current for each trial is saved
%                 as trial current file. 
%  dsampf       : <optional> <<int>> Specify frequency of
%                 downsampling. This value must be smaller than the
%                 original sampling frequency of M/EEG data. 
%  tsubsmpl     : <obsolete> <optional> <<int vector>> Specify subsampled
%                 time index. If 'tsubsmpl' is empty or not given, time
%                 subsampling is not done. Instead of this, 'dsampf'
%                 parameter is recommended for downsampling. 
%  overlap_mode : <optional> <<bool>> If false (default), cortical
%                 current is averaged over overlapped time
%                 windows. Otherwise, current is not averaged over
%                 overlapped time windows; current time series of each
%                 time windows are concatenated sequentially for spectral
%                 analysis. 
%  If following fields are given, these values are used instead of
%  bayes_parm field in result file (see vb_job_vb): 
%   basisfile
%   megfile  
%   twin_meg
%   Tperiod  
%   Tnext
%   extra.basisfile (for extra dipole)
% ---
%
% [output]
% Zact : <<float matrix>> Z-current.
%  Zact(n,t,:) is the current at the vertex 'ix_act(n)' & the time 't'. 
%  Zact(Nact,Nsample)          for trial_average = true. 
%  Zact(Nact,Nsample,Ntrials)  for trial_average = false. 
%   Nact     : # of active region, 
%   Nsample  : # of time sample, 
%   Ntrials  : # of trials in all session]
% Jinfo: <<struct>> Information of cortical current.
% Jext : <<double matrix>> Extra dipole current. 
%
% [example]
% >> parm.areafile = './NG/data/100009d.area.mat';
% >> parm.bayesfile = './NG/result/result3_col.bayes.mat';
% >> parm.trial_average = OFF;
% >> parm.area_key = 'fMTl';
% >> parm.currfile = './result/result3_col_fMTl_1.curr.mat';
% >> vb_job_current(parm);
%
% 2005-03-31 Taku Yoshioka
% 2005/08/22 O. Yamashita ver.30b
% 2005/08/24 O. Yamashita Modified
% 2006/03/24 M. Sato Modified
%   Eliminate zero variance point in current reconstruction
% 2006/04/06 D. Kawawaki modified  
%   'project root' problem was solved.
% 2006/07/19 O. Yamashita Modified
% 2006/09/01 M. Sato
% 2008/07/01 M. Sato
%  alpha is scaled back by bsnorm before saving in new vb_job_vb
%  Then, current is calculated by using leadfield without normalization
%  Leadfield different from that used in VB estimation can be used
% 2008-08-19 Taku Yoshioka
%  Extended dipole was supported. 
% 2009-04-02 Taku Yoshioka
%  Parameter name changed within this code for readability
%  (just replacing 'resultfile' to bayesfile)
% 2009-03-01 rhayashi
%  bugfix(ix_area was not given to vbmeg_reconstruct functions)
% 2010-12-03 taku-y
%  [trivial] Comment modified.
%  [trivial] Struct 'curr_parm' available without field 'areafile'. 
% 2010-12-06 taku-y
%  [trivial] Comment modified.
% 2011-05-11 taku-y
%  [minor] 'vbmeg_current_reconstruct_z(_tr)' was replaced by
%  'vb_current_reconstruct_z(_tr)', which is the same excepting
%  Jinfo.Tmsec corresponds to overlaped time windows.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% check input arguments
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if length(varargin) == 1
  proj_root = [];
  curr_parm = varargin{1};
elseif length(varargin) == 2
  proj_root = varargin{1};
  curr_parm = varargin{2};
else
  error('Error: invalid usage of job_current.');
end

proj_root = vb_rm_trailing_slash(proj_root);

if ~isfield(curr_parm,'trial_average'), 
  curr_parm.trial_average = true; 
end

if ~isfield(curr_parm,'areafile'), 
  curr_parm.areafile = [];
end

%
% set absolute path
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isempty(proj_root)
  bayesfile = [proj_root filesep curr_parm.bayesfile]; % bayes file
  currfile  = [proj_root filesep curr_parm.currfile];  % current file
  if isfield(curr_parm, 'areafile')
    areafile   = [proj_root filesep curr_parm.areafile];   % vb_get_area
  end
else
  bayesfile = curr_parm.bayesfile;
  currfile  = curr_parm.currfile;
  if isfield(curr_parm, 'areafile')
    areafile   = curr_parm.areafile;
  end
end

%
% Current estimation
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp('Start current estimation')

if ~isempty(areafile) ...
  & isfield(curr_parm, 'area_key') & ~isempty(curr_parm.area_key),
    Area = vb_get_area(areafile, curr_parm.area_key);
    ix_area = Area.Iextract;
else
    ix_area = [];
end

parm = curr_parm;
parm.ix_area = ix_area;
if curr_parm.trial_average == false ...
      && isfield(curr_parm,'jactdir') && ~isempty(curr_parm.jactdir)
  [Zact, Jinfo, bayes_parm, vb_parm, MEGinfo, Jext, Pointlist] ...
      = vb_current_reconstruct_z_tr(proj_root, parm);
else
  [Zact, Jinfo, bayes_parm, vb_parm, MEGinfo, Jext, Pointlist] ...
      = vb_current_reconstruct_z(proj_root, parm);
end;

% load brain vertex number
brain_file = fullfile(proj_root, bayes_parm.brainfile);
Nvertex = vb_load_cortex_info(brain_file);


%
% Save result
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp(sprintf('--- Save estimated current: %s\n',currfile));
vb_fsave(currfile,'Zact','Jinfo','MEGinfo','bayes_parm','vb_parm',...
         'Jext','Pointlist', 'curr_parm', 'Nvertex');

%
% project_file save
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
proj_file = get_project_filename;
if isempty(proj_file)
  return;
end

project_file_mgr('load', proj_file);
project_file_mgr('add', 'current_parm', curr_parm);

return;
