function job_plot_meg_load_meg(inst_id,megfile,mode,twin_lim,twin_ini)
% Load MEG data (callback function of job_plot_meg).
%
% [syntax]
% job_plot_meg_load_meg(inst_id,megfile,mode,twin_lim,twin_ini)
% [input]
% - inst_id       : instance id of gui.
% - megfile       : MEG data file (.meg.mat)
% - mode          : = 0 % loading average data
%                   = 1 % loading all trials
%                   = 2 % intaraction mode(default)
% - twin_lim      : Time window selection limit(meg data index)
%                   e.g. twin_lim = [300, 500];
% - twin_ini      : Initial Time window selection(meg data index)
%                   (default:[Pretrigger, Pretrigger] from 0msec to 0msec)
%                   e.g. twin_ini = [300, 500];
%
% [history]
% 2008-09-02 Taku Yoshioka
% 2008-10-24 Taku Yoshioka
%   Support EEG data
% 2011-02-10 taku-y
%  [debug] Channelinfo correctly loaded.
% 2011-06-01 (Sako) converted return values of vb_load_device to upper case
% 2012-06-12 taku-y
%  [minor] Not invoke confirmation dialog for trial averaging when
%  MEG-MAT file contains single trial. 
% 2013-7-05  Masa-aki Sato 
%    load sensor pos using 'vb_load_channel_pos'
% 2015-12-14 rhayashi
%    supported fileinfo MEG/EEG-MAT format.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

global vbmeg_inst;
define_job_plot_meg_const;

%
% Input parameters
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if nargin<5, twin_ini = []; end
if nargin<4, twin_lim = []; end
if nargin<3, mode = 2; end

%
% Check file and switch average/raw data
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~exist(megfile,'file'),
  warndlg(['File ' megfile ' does not exist.']);
  return;
end

if mode == 2, 
  MEGinfo = vb_load_meg_info(megfile);
  if MEGinfo.Nrepeat>1, 
    button = questdlg(['Which of average or raw MEG/EEG data ' ...
                       'do you want to load?'],'Load MEG/EEG data', ...
                      'Average','Raw','Average');
    if strcmp(button,'Average'), 
      mode = 0; 
    else 
      mode = 1; 
    end
  else
    mode = 0;
  end
end

%
% Load MEG data
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
set(vbmeg_inst.plotmeg{inst_id}.h_fig,'Pointer','watch');
drawnow;

% MEG data
pick = vb_load_channel(megfile);
if isempty(pick)
    job_plot_meg_close(inst_id);
    error('Specified MEG/EEG file doesn''t contain sensor position. Import data with .pos.mat file.');
    return;
end
measurement_str = vb_load_device(megfile);
if strcmpi(measurement_str, 'INFO')
    load(megfile, 'fileinfo');
    measurement_str = vb_load_device(fileinfo.filename{1});
end
Measurement_str = upper(measurement_str);
[bexp, ch_info] = vb_load_meg_data(megfile);
MEGinfo = vb_load_meg_info(megfile);

% Channel type information
MEGinfo.ChannelInfo = ch_info;

if mode==0, bexp = mean(bexp,3); end
pretrigger = vb_meginfo_get_pre_trigger(MEGinfo);
if isempty(pretrigger), pretrigger = 0; end
N = size(bexp,1);
T = ((1:MEGinfo.Nsample)-pretrigger-1)*(1000/MEGinfo.SampleFreq);
if isempty(twin_lim), twin_lim = [1, MEGinfo.Nsample]; end
if isempty(twin_ini), twin_ini = [1, 1]; end

vbmeg_inst.plotmeg{inst_id}.megfile = megfile;
% Measurement_str = vb_load_device(megfile);
switch Measurement_str
 case 'MEG', 
  vbmeg_inst.plotmeg{inst_id}.bexp = bexp*1e15; % T -> fT
  vbmeg_inst.plotmeg{inst_id}.ylabel_str = 'MEG signal [fT]';
 case 'EEG',
  vbmeg_inst.plotmeg{inst_id}.bexp = bexp*1e6; % V -> micro V
  vbmeg_inst.plotmeg{inst_id}.ylabel_str = 'EEG signal [\mu V]';
end

clear bexp;
vbmeg_inst.plotmeg{inst_id}.pick = pick;
vbmeg_inst.plotmeg{inst_id}.MEGinfo = MEGinfo;
vbmeg_inst.plotmeg{inst_id}.N = N; 
vbmeg_inst.plotmeg{inst_id}.T = T;
vbmeg_inst.plotmeg{inst_id}.twin_lim = twin_lim;
vbmeg_inst.plotmeg{inst_id}.twin_ini = twin_ini;

% Display parameters
vbmeg_inst.plotmeg{inst_id}.n_trial = 1; 
vbmeg_inst.plotmeg{inst_id}.t = [T(min(twin_ini)) T(max(twin_ini))]; 

%
% Delete old menu for sensor type
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
h = vbmeg_inst.plotmeg{inst_id}.sensor_type_menu;
hh = get(h,'Children');
for i=1:length(hh)
  delete(hh(i));
end

%
% Create new menu for sensor type
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

sensor_type_list   = [];
sensor_type_string = cell(0);
if strcmpi(Measurement_str, 'MEG') && ~isempty(ch_info)

    % Sensor type information for Yokogawa MEG (@ATR)
    meg_sensor_type_list = sort(unique(ch_info.Type));

    for k=1:length(meg_sensor_type_list)
        sensor_type = meg_sensor_type_list(k);
        if sensor_type == 1
            sensor_type_list = [sensor_type_list, 1];
            sensor_type_string{k} = 'Magnetometer';
        elseif sensor_type == 2
            sensor_type_list = [sensor_type_list, 2];
            sensor_type_string{k} = 'Axial gradiometer';
        elseif sensor_type == 3
            sensor_type_list = [sensor_type_list, 3];
            sensor_type_string{k} = 'Planar gradiometer';
        end
    end
end
sensor_type_list = [sensor_type_list, -1]; % -1 is all;
sensor_type_string{end+1} = 'All';

% Create menu
vbmeg_inst.plotmeg{inst_id}.sensor_type_list = sensor_type_list;
h = vbmeg_inst.plotmeg{inst_id}.sensor_type_menu;

for i=1:length(sensor_type_list)
  command = ['job_plot_meg_change_sensor_type(' num2str(inst_id) ...
             ',' num2str(i) ');'];
  hh = uimenu(h,'Label',sensor_type_string{i},'Callback',command, ...
              'Tag',['submenu' num2str(i)]);
  vbmeg_inst.plotmeg{inst_id}.h_sensortype(i) = hh;
  set(hh,'Checked','off');
end

% Set sensor type (default:Axial gradiometer)
ix = find(sensor_type_list == 2);
if ~isempty(ix)
    vbmeg_inst.plotmeg{inst_id}.sensor_type = 2;  % Axial gradiometer
    set(vbmeg_inst.plotmeg{inst_id}.h_sensortype(ix),'Checked','on');
else
    vbmeg_inst.plotmeg{inst_id}.sensor_type = -1; % All
    set(vbmeg_inst.plotmeg{inst_id}.h_sensortype(end),'Checked','on');
end

% Reset sensor infomation
job_plot_meg_choose_plot_sensor(inst_id, 'reset');

%
% Update plots
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
job_plot_meg_update_spatialpattern(inst_id);
job_plot_meg_update_timecourse(inst_id,FORCE_UPDATE); 

set(vbmeg_inst.plotmeg{inst_id}.h_fig,'Pointer','arrow');

%
% Change figure title
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[tmp,fname,ext] = fileparts(megfile);
set(vbmeg_inst.plotmeg{inst_id}.h_fig,'Name', ...
                  ['job_plot_meg: ' fname ext]);
