function [data] = vb_trial_extractor_output_file(data)
% Create trial MEG/EEG-MAT file.
% [USAGE]
%    [data] = vb_trial_extractor_output_file(data);
% [IN]
%    data : vb_trial_extractor object.
% [OUT]
%    data : vb_trial_extractor object.
% [see also]
%    vb_msrmnt_make_trial_data
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('data', 'var')
    error('data is a required parameter.');
end

%
% --- Main Procedure
%

% Basename definition
org_file = vb_continuous_file_get_filename(data.continuous_file);
base_extension         = org_file(end-8+1:end);
[base_path, base_file] = vb_get_file_parts(org_file(1:end-8));

sample_freq = vb_continuous_file_get_sample_freq(data.continuous_file);

Nputfile = length(data.output_file);

proc_spec = struct;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ORIGINAL FILE INFO
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
proc_spec.proc_parm = struct;
proc_spec.proc_parm.sample_freq   = sample_freq;
data_file = org_file;

% If downsampling is specified
if ~isfield(data, 'output_sample_freq')
    data.output_sample_freq = sample_freq;
end

if data.output_sample_freq ~= sample_freq
    % Create down sampling file
    dsfile = fullfile(base_path, [base_file, '_', num2str(data.output_sample_freq), 'Hz', base_extension]);
    vb_disp(sprintf('Now creating downsampling file : %s\n', ds_file));
    if isfield(data, 'batch_mode') && data.batch_mode
        h = -1;
    else
        h = msgbox('Now creating downsampling file...', 'Please wait');
        bh = findall(h, 'Style', 'pushbutton');
        set(bh, 'Visible', 'off'); drawnow; pause(0.1);
    end
    if ishandle(h), delete(h), end
    err = inner_make_down_sample_file(org_file, dsfile, data.output_sample_freq);
    if ~err
        % Create success
        proc_spec.proc_parm.sample_freq = data.output_sample_freq;
        data_file   = dsfile;
    else
        % Create failure
        errmsg = sprintf('Unable to make down sampling file : %s\n', dsfile);
        if isfield(data, 'batch_mode') && data.batch_mode
            error(errmsg);
        else
            errordlg(errmsg, 'Output downsampling file');
            return;
        end
    end
end

for k=1:Nputfile
    % vb_msrmnt_make_trial_data
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Output TRIAL INFO
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    proc_spec.proc_parm.Pretrigger_ms  = data.trial_setting.pretrigger_ms;
    proc_spec.proc_parm.Posttrigger_ms = data.trial_setting.posttrigger_ms;

    % tag
    tag_obj = vb_tag_util_find_tag(data.label_list, data.output_file{k}.label_name);
    trial_ix = vb_tag_get_trial_index(tag_obj);

    % get onset sample list
    proc_spec.proc_parm.trigger = sort([data.trial_list(trial_ix).onset]);
    % Round trigger point
    proc_spec.proc_parm.trigger = round(...
                                      proc_spec.proc_parm.trigger * ...
                                     (data.output_sample_freq/sample_freq));

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % OUTPUT FILENAME
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    new_file = data.output_file{k}.output_file;

    tmp = new_file;
    try
        vb_save(tmp, 'tmp');
        delete(tmp);
    catch
        errordlg(['Unable to make file : ', tmp, ...
                  sprintf('\nCheck directory permission.'), ...
                 ], 'Output MEG(EEG) file');
        return;
    end
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Start creating file
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    if isfield(data, 'batch_mode') && data.batch_mode
        h = -1;
        vb_disp('Now putting file(s). Please wait');
    else
        h = msgbox('Now putting file(s) ...', 'Please wait');
        bh = findall(h, 'Style', 'pushbutton');
        set(bh, 'Visible', 'off'); drawnow; pause(0.1);
    end

    vb_disp_nonl(sprintf('Label: %s -> %s ... \n',...
                    vb_tag_get_tag_name(tag_obj), ...
                    new_file));

    vb_msrmnt_make_trial_data(data_file, proc_spec, new_file);
    vb_disp('done.');
    if ishandle(h), delete(h); end
end


function [err] = inner_make_down_sample_file(org_file, new_file, freq)
err = false;

try
MEGinfo = vb_load_measurement_info(org_file);
measurement = vb_info_get_measurement(MEGinfo);

switch(upper(measurement))
    case 'MEG'
        func_name = 'vb_megfile_filter_ch_data';
    case 'EEG'
        func_name = 'vb_eegfile_filter_ch_data';
    otherwise
        err = true;
        error('Unknown measurement device was specified.');
end

proc_spec.parm.fsamp = freq;
feval(func_name, org_file, proc_spec, new_file);

catch
    err = true;
end
