function [is_valid, reason] = vb_signal_processor_is_valid_parm(parm)
% check the validity of a specified parmeter.
% [USAGE]
%    [is_valid, reason] = vb_signal_processor_is_valid_parm(parm);
% [IN]
%    parm : parameter.
% [OUT]
%    is_valid : true/false. true means that the parameter is valid. [bool]
%      reason : error reason [string]
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('parm', 'var')
    error('parm is a required parameter.');
end

%
% --- Main Procedure
%
d = vb_define_signal_processor;

switch(parm.type)
    case d.TYPE_PROCESSING_BIAS_CORRECTION
        [is_valid, reason] = inner_check_bias_correction(parm);
    case d.TYPE_PROCESSING_HIGHPASS        
        [is_valid, reason] = inner_check_highpass(parm);
    case d.TYPE_PROCESSING_LOWPASS         
        [is_valid, reason] = inner_check_lowpass(parm);
    case d.TYPE_PROCESSING_BANDPASS        
        [is_valid, reason] = inner_check_bandpass(parm);
    case d.TYPE_PROCESSING_STOPBAND        
        [is_valid, reason] = inner_check_stopband(parm);
    case d.TYPE_PROCESSING_DOWNSAMPLE      
        [is_valid, reason] = inner_check_downsampling(parm);
    case d.TYPE_PROCESSING_COMMON_REFERENCE
        [is_valid, reason] = inner_check_common_reference(parm);
    otherwise
        error('Unknown parameter type was specified.');
end

%
% --- Check parameter section
%

function [is_valid, reason] = inner_check_bias_correction(parm)
is_valid = true;
reason = '';
if length(parm.mode) == 2
    if parm.mode(1) < 0 || parm.mode(2) <= 0 || parm.mode(1) >= parm.mode(2)
        is_valid = false;
        reason = 'Specified ''mode''(time window[t1, t2]) is invalid.';
    end
elseif (parm.mode == 0 || parm.mode == 1 || parm.mode == 2)
    % do nothing
else
    is_valid = false;
    reason   = 'The specified ''mode'' is invalid.';
end


function [is_valid, reason] = inner_check_highpass(parm)
is_valid = true;
reason = '';
if parm.cutoff_freq <= 0
    is_valid = false;
    reason   = 'The specified ''Cutoff frequency'' is invaild.';
elseif ~(parm.impulse_response == 1 || parm.impulse_response == 2)
    is_valid = false;
    reason   = 'The specified ''Impulse response'' is invalid.';
elseif parm.impulse_response == 2 && ...
       (parm.order < 0 || ceil(parm.order) ~= parm.order)
        is_valid = false;
        reason = 'The specified ''Filter order'' is invalid.';
elseif ~(parm.filtfilt == 0 || parm.filtfilt == 1)
    is_valid = false;
    reason   = 'The specified ''Time delay correction'' is invalid.';
end

function [is_valid, reason] = inner_check_lowpass(parm)
is_valid = true;
reason = '';
[is_valid, reason] = inner_check_highpass(parm);


function [is_valid, reason] = inner_check_bandpass(parm)
is_valid = true;
reason = '';
if parm.cutoff_freq_low <= 0
    is_valid = false;
    reason   = 'The specified ''Lower cutoff frequency'' is invalid.';
elseif parm.cutoff_freq_high <= 0
    is_valid = false;
    reason   = 'The speicifed ''Higher cutoff frequency'' is invaild.';
elseif parm.cutoff_freq_low >= parm.cutoff_freq_high
    is_valid = false;
    reason   = 'The specified ''Lower cutoff frequency'' is larger than ''Higher frequency''.';
elseif ~(parm.impulse_response == 1 || parm.impulse_response == 2)
    is_valid = false;
    reason   = 'The specified ''Impulse response'' is invalid.';
elseif parm.impulse_response == 2 && ...
       (parm.order < 0 || ceil(parm.order) ~= parm.order)
        is_valid = false;
        reason = 'The specified ''Filter order'' is invalid.';
elseif ~(parm.filtfilt == 0 || parm.filtfilt == 1)
    is_valid = false;
    reason   = 'The specified ''Time delay correction'' is invalid.';
end

function [is_valid, reason] = inner_check_stopband(parm)
is_valid = true;
reason = '';
[is_valid, reason] = inner_check_bandpass(parm);

function [is_valid, reason] = inner_check_downsampling(parm)
is_valid = true;
reason = '';
if parm.dsamp_freq < 1 || (ceil(parm.dsamp_freq) ~= parm.dsamp_freq)
    is_valid = false;
    reason   = '''New sampling frequency'' should be integer larger than 1.';
end

function [is_valid, reason] = inner_check_common_reference(parm)
is_valid = true;
reason = '';
if ~(parm.mode == 0 || parm.mode == 1)
    is_valid = false;
    reason   = 'The specified ''mode'' is invalid.';
end
