function [varargout] = gain_input(varargin)
% This gui is for input eeg gain values.
% When the figure close, eeg gain values are returned.
%
% Usage : 
% [eeg_gain, cancelled] = gain_input('init', <channel_list> [,gain_list]);
%
% [IN]
%            key  : 'init', 'cancel', 'callback'
%    channel_list : Nx1 channel numbers.
%    gain_list    : Nx1 gain initial values.(This is optional input)
%
% [OUT]
%    gain : inputted gain value(Nx2)
%            gain{:, 1} = channel list{Nx1}
%            gain{:, 2} = gain list   {Nx1}
%    cancelled : true  % gui closed by pushing cancel or x button.
%                false % gui close by pushing ok button.
% Note: inner expression of gain value, 
%       gain value = 0 means not inputted gain.
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)
if nargin < 1
    key = 'init';
else
    key = varargin{1};
end

switch key
    case 'init'
        gain = [];
        if nargin == 2
            channel_list = varargin{2};
            gain_list = zeros(length(channel_list), 1);
        elseif nargin == 3
            channel_list = varargin{2};
            gain_list    = varargin{3};
            
            if length(channel_list) ~= length(gain_list)
                error('Number of channels and gain values mismatched.');
            end
            cl_size = size(channel_list);
            gl_size = size(gain_list);
            if cl_size(2) > 1
                warning('channel_list size should be Nx1.');
                channel_list = channel_list';
            end
            if gl_size(2) > 1
                warning('gain_list size should be Nx1.');
                gain_list = gain_list';
            end
        else
            error(USAGE_STRING);
        end

        % gain setting
        Nchannel = length(channel_list);
        gain = cell(Nchannel, 2);
        gain(:, 1) = channel_list;
        for k=1:Nchannel
            gain{k, 2} = gain_list(k);
        end

        h = create_figure(gain);
        % Wait for application end
        waitfor(h, 'Name', 'close');
        if ~ishandle(h)
            % close by cancel or x button
            varargout{1} = [];
            varargout{2} = true;
            return;
        end

        if get_application_status(h) == VALUE_RETURN_EXIT
            [gain_data] = get_application_data(h);
            % Remove not inputted(=0) channel gain.
            null_ix = find([gain_data{:, 2}]  == 0);
            gain_data(null_ix, :) = [];
            varargout{1} = gain_data;
            varargout{2} = false;
        elseif get_application_status(h) == VALUE_NO_RETURN_EXIT
            varargout{1} = [];
            varargout{2} = true;
        else
            error('Assertion.', 'error');
        end
        % delete figure
        if ishandle(h)
            delete(h);
        end
    case 'cancel'
        set_application_status(gcf, VALUE_NO_RETURN_EXIT);
        closereq;
    case 'callback'
        callback(gcf, varargin{2});
end

function h = create_figure(gain)
% This function create main figure.
% [IN]
%    gain : gain(:, 1) % Nx1 channel list
%           gain(:, 2) % Nx1 gain list
% [OUT]
%    h : created figure handle

    h = openfig(mfilename);
    create_item(h, gain);
    set_application_data(h, gain);
    set_application_status(h, VALUE_NO_RETURN_EXIT);

function create_item(fig, gain)
% This function creates editboxes and set initial gain value.
%
% [IN]
%     fig : figure handle
%    gain : gain(:, 1) % Nx1 channel list
%           gain(:, 2) % Nx1 gain list
%

    H = guihandles(fig);

    channels  = gain(:, 1);
    gain_list = gain(:, 2);

    HORIZONTAL_ITEM_NUM = ceil(length(channels) / VERTICAL_ITEM_NUM);

    % Figure size setting
    if HORIZONTAL_ITEM_NUM >= 4
        size_f = get(fig, 'Position');
        size_f(3) = size_f(3) * 2;
        set(fig, 'Position', size_f);
    end
    
    % Unit setting
    set(H.unit_text, 'String', UNIT_STRING);
        
    % Frame setting
    FRAME_LEFT_MARGIN = 0.05;
    FRAME_RIGHT_MARGIN = 0.05;
    FRAME_TOP_MARGIN  = 0.1;
    FRAME_BOTTOM_MARGIN = 0.1;

    set(H.frame, 'Position', ...
        [FRAME_LEFT_MARGIN, FRAME_BOTTOM_MARGIN, ...
        1.0-(FRAME_LEFT_MARGIN + FRAME_RIGHT_MARGIN), ...
        1.0-(FRAME_TOP_MARGIN + FRAME_BOTTOM_MARGIN)]);


    % Item area(=Frame inner) setting
    FRAME_INNER_LEFT_MARGIN = 0.02;
    FRAME_INNER_RIGHT_MARGIN = 0.02;
    FRAME_INNER_TOP_MARGIN   = 0.05;
    FRAME_INNER_BOTTOM_MARGIN = 0.05;

    FRAME_INNER_WIDTH  = 1.0 - ...
                        (FRAME_LEFT_MARGIN+FRAME_RIGHT_MARGIN+ ...
                         FRAME_INNER_LEFT_MARGIN+FRAME_INNER_RIGHT_MARGIN);
    FRAME_INNER_HEIGHT = 1.0 - ...
                        (FRAME_TOP_MARGIN+FRAME_BOTTOM_MARGIN+ ...
                         FRAME_INNER_TOP_MARGIN+FRAME_INNER_BOTTOM_MARGIN);

    % Item Setting
    ITEM_WIDTH  = FRAME_INNER_WIDTH / HORIZONTAL_ITEM_NUM;
    ITEM_HEIGHT = FRAME_INNER_HEIGHT / VERTICAL_ITEM_NUM;

    CH_TEXT_WIDTH        = ITEM_WIDTH * 0.3; 
    GAIN_EDIT_WIDTH      = ITEM_WIDTH * 0.3;
    GAIN_UNIT_TEXT_WIDTH = ITEM_WIDTH * 0.4;
    
    MAX_CH_TEXT_WIDTH = 0.1;
    MAX_GAIN_EDIT_WIDTH = 0.08;
    MAX_GAIN_UNIT_TEXT_WIDTH = 0.10;
    
    if CH_TEXT_WIDTH > MAX_CH_TEXT_WIDTH
        CH_TEXT_WIDTH = MAX_CH_TEXT_WIDTH;
    end
    if GAIN_EDIT_WIDTH > MAX_GAIN_EDIT_WIDTH
        GAIN_EDIT_WIDTH = MAX_GAIN_EDIT_WIDTH;
    end
    if GAIN_UNIT_TEXT_WIDTH > MAX_GAIN_UNIT_TEXT_WIDTH
        GAIN_UNIT_TEXT_WIDTH = MAX_GAIN_UNIT_TEXT_WIDTH;
    end

    %
    % --- Create edit components
    %
    itemNum = 0;

    ITEM_LEFT_BEGIN = FRAME_LEFT_MARGIN + FRAME_INNER_LEFT_MARGIN;
    ITEM_BOTTOM_BEGIN  = 1.0 - (FRAME_TOP_MARGIN + FRAME_INNER_TOP_MARGIN);

    for k=1:HORIZONTAL_ITEM_NUM
        for j=1:VERTICAL_ITEM_NUM
            ITEM_LEFT   = ITEM_LEFT_BEGIN + (k-1) * ITEM_WIDTH;
            ITEM_BOTTOM = ITEM_BOTTOM_BEGIN - (j * ITEM_HEIGHT);
            % Channel name
            channel = channels{itemNum+1};
            if isnumeric(channel)
                channel = num2str(channel);
            end
            uicontrol('String', ['ch:' channel ' '], 'style', 'text', 'units', 'normalized', ...
                        'HorizontalAlignment', 'right', ...
                        'parent', fig,...
                        'position',[ITEM_LEFT, ITEM_BOTTOM, ...
                                    CH_TEXT_WIDTH, ITEM_HEIGHT]);
            % Gain input edit box
            gain_str = num2str(gain_list{itemNum+1});
            if strcmp(gain_str, '0')
                gain_str = '';
            end
            uicontrol( 'style', 'edit', 'units', 'normalized', ...
                        'HorizontalAlignment', 'right', ...
                        'parent', fig,...
                        'String', gain_str,...
                        'Tag', [channel '_edit'],...
                        'TooltipString', GAIN_EDIT_TOOL_TIP_STRING,...
                        'callback', 'gain_input(''callback'', gcbo)',... 
                        'position',[ITEM_LEFT+CH_TEXT_WIDTH, ITEM_BOTTOM, ...
                                    GAIN_EDIT_WIDTH, ITEM_HEIGHT]);
            % [uV/V] text
            uicontrol('String', [' ', UNIT_STRING], 'style', 'text', 'units', 'normalized', ...
                        'HorizontalAlignment', 'left', ...
                        'parent', fig,...
                        'position',[ITEM_LEFT+CH_TEXT_WIDTH+GAIN_EDIT_WIDTH, ITEM_BOTTOM, ...
                                    GAIN_UNIT_TEXT_WIDTH, ITEM_HEIGHT]);
            itemNum = itemNum + 1;
            if itemNum >= length(channels)
                break;
            end
        end
    end

function callback(h, hObj)
% This function processes user input
%
% [IN]
%       h : figure handle
%    hObj : component handle(action object handle)
%

    H = guihandles(h);

    switch(hObj)
        case H.input_all_push
            gain_value = str2num(get(H.input_all_edit, 'String'));
            [result, err_msg] = Is_gain_value_OK(gain_value);
            if result == false
                errordlg(err_msg, 'error');
                return;
            end
            res = questdlg('Input gain to all channels?', 'confirm', 'Yes', 'No', 'Yes');
            if strcmp(res, 'Yes')
                gain_all_str = get(H.input_all_edit, 'String');
                h_editboxes = findobj(h, 'ToolTipString', GAIN_EDIT_TOOL_TIP_STRING);
                set(h_editboxes, 'String', gain_all_str);
            end
        case H.input_all_edit
%             gain_value = str2num(get(H.input_all_edit, 'String'));
%             if Is_gain_value_OK(gain_value)
%                 set(H.input_all_push, 'Enable', 'on');
%             else
%                 set(H.input_all_push, 'Enable', 'off');
%             end
        case H.ok_push
            [eeg_gain] = get_gain_data_from_editbox(h);
%            [result, ch_no] = has_empty_input(eeg_gain);
            result = false;
            if result
                errordlg(['ch', num2str(ch_no(1)), ' : gain value is empty.'], 'error');
            else
                set_application_status(h, VALUE_RETURN_EXIT);
                set_application_data(h, eeg_gain);
                escape_waitfor(h);
            end
        case H.cancel_push
            set_application_status(h, VALUE_NO_RETURN_EXIT);
            closereq;
        otherwise
            % This is gain edit box callback
            str_tool_tip = get(hObj, 'ToolTipString');
            if strcmp(str_tool_tip, GAIN_EDIT_TOOL_TIP_STRING)
                str_tag = get(hObj, 'Tag');
                [channel, count, errmsg] = sscanf(str_tag, '%[^_]_edit');
                if ~isempty(errmsg)
                    error('gain_input application error.');
                end
                gain_value = str2num(get(hObj, 'String'));
                [result, err_msg] = Is_gain_value_OK(gain_value);
                if result == false
                    % Invalid value
                    errordlg(['ch' num2str(channel) ' : ', err_msg], 'error');
                    set(hObj, 'String', '');
                else
                    set(hObj, 'String', num2str(gain_value));
                end
            end
    end

function [result, err_msg] = Is_gain_value_OK(gain_value)
% This function checks gain value which is collect or not.
%
% [IN]
%    gain_value : gain value.
% [OUT]
%    result : true  % gain value is collect.
%             false % gain value is illigal.
%    err_msg: illigal reason.
%
   result = false;
   err_msg = [];

%   if isempty(gain_value)
%       err_msg = 'Inputted value is invalid.';
%   else
   if gain_value == 0
       err_msg = '0 cannot be specified.';
   else
       result = true;
   end

function escape_waitfor(h)
% Call this function, escape 'waitfor' loop in 'init' processing.
% In 'init' processing, use 'waitfor' MATLAB function.
% [IN]
%    fig : figure handle

    set(h, 'Name', 'close');  % Waitfor exit by this changing.

function set_application_status(h, status)
% This function sets application status
% [IN]
%         h : figure handle
%    status : VALUE_RETURN_EXIT or VALUE_NO_RETURN_EXIT

    % load current data
    s = guidata(h);

    % set status
    s.status = status;

    % save data
    guidata(h, s);

function [status] = get_application_status(h)
% This function returns application status.
% [IN]
%    h : figure handle
% [OUT]
%    status : VALUE_RETURN_EXIT or VALUE_NO_RETURN_EXIT

    % load current data
    s = guidata(h);

    % return status
    status = s.status;

function set_application_data(h, gain_data)
% This function sets application data to figure.
%
% [IN]
%            h : figure handle
%    gain_data : gain_data(:, 1) : Nx1 channel list
%                gain_data(:, 2) : Nx1 gain value list

    % load current data
    s = guidata(h);

    % set gain_data
    s.data.gain_data = gain_data;
    
    % save data
    guidata(h, s);

function [gain_data] = get_application_data(h)
% This function returns application data.
%
% [IN]
%    h : figure handle
% [OUT]
%    gain_data : gain_data(:, 1) : Nx1 channel list
%                gain_data(:, 2) : Nx1 gain value list
%
    % load current data
    s = guidata(h);

    % set gain data
    gain_data = s.data.gain_data;

function [gain] = get_gain_data_from_editbox(h)
% This function returns gain data which editboxes have.
%
% [IN]
%    h : figure handle
% [OUT]
%    gain : gain data
%
%
    [gain] = get_application_data(h);

    for k=1:length(gain(:, 1))
        % get editbox object
        child_h = findobj(h, 'Tag', [num2str(gain{k, 1}) '_edit']);

        % get gain value
        value = str2num(get(child_h, 'String'));
        if ~isempty(value)
            gain{k, 2} = value;
        else
            gain{k, 2} = 0;
        end
    end

function [result, ch_no] = has_empty_input(gain)
% This function check gain array and 
% returns channel number which is not inputted gain.
% [IN]
%    gain : gain(:, 1) Nx1 channel number list
%           gain(:, 2) Nx1 gain list
% [OUT]
%    result : true  % There is empty input.
%             false % There is no empty input.
%     ch_no : empty channel number.

    result = false;
    ch_no  = [];

    for k=1:length(gain(:, 1))
        if gain(k, 2) == 0
            ch_no = [ch_no;gain(k, 1)];
            result = true;
        end
    end

    ch_no = ch_no';

%
% --- Constant Definition
%
function [value] = VERTICAL_ITEM_NUM
    value = 20;
function [value] = VALUE_RETURN_EXIT
    value = 0;
function [value] = VALUE_NO_RETURN_EXIT
    value = 1;
function [str] = GAIN_EDIT_TOOL_TIP_STRING
    str = 'gain';
function [str] = UNIT_STRING
    str = '[V/V]';
function [str] = USAGE_STRING
    str = 'Usage : [eeg_gain, cancelled] = gain_input(''init'', channel_list [,gain_list]);';
