function [obj, result] = ...
             bm_fileio_import_file(obj, file_type, model_file, analyze_info, args)
% import brain model
% [USAGE]
%    [obj] = bm_fileio_import_file(<obj>, <file_type>, <model_file>, <analyze_info>);
% [IN]
%             obj : bm_fileio object
%       file_type : import file type
%      model_file : brain model(.brain.mat or .head.mat) filename
%    analyze_info : analyze_info object(To convert coordinate system)
%            args : optional parameters.
%
% [OUT]
%       obj : bm_fileio object
%    result : brain model import result.
%             (SUCESS or FAILURE)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('obj', 'var'), error('obj is a required parameter.'); end
if ~strcmp(obj.class_type, 'bm_fileio')
    error('class type is invalid.');
end
if ~exist('file_type', 'var')
    error('file_type is a required parameter.');
end
if ~exist('model_file', 'var')
    error('model_file is a required parameter.');
end
if ~exist('analyze_info', 'var')
    error('analyze_info is a required parameter.');
end

%
% --- Main Procedure
%
global vbmeg_inst;
result = SUCCESS;

app_const = define_bm_edit_app;

% read brain model
switch(file_type)
    case app_const.file_type.BRAIN_FILE
        [V_cel, F_cel, xx_cel, name_cel, conduct_cel] = ...
            bm_fileio_read_brain_file(model_file, args{1}, analyze_info);
    case app_const.file_type.HEAD_FILE
        [V_cel, F_cel, xx_cel, name_cel, conduct_cel] = ...
            bm_fileio_read_head_file(model_file);
    case app_const.file_type.CURRY_FILE
        [V_cel, F_cel, xx_cel, name_cel, conduct_cel] = ...
            bm_fileio_read_curry_file(model_file, analyze_info);
    case app_const.file_type.FACE_FILE
        [V_cel, F_cel, xx_cel, name_cel, conduct_cel] = ...
            bm_fileio_read_face_file(model_file);
    case app_const.file_type.MASK_FILE
        [V_cel, F_cel, xx_cel, name_cel, conduct_cel] = ...
            bm_fileio_read_mask_file(model_file, args{1}, analyze_info);
    case app_const.file_type.SPM_NORM_FILE
        [V_cel, F_cel, xx_cel, name_cel, conduct_cel] = ...
            bm_fileio_read_spm_norm_file(model_file, args{1}, analyze_info);
    otherwise
        error('specified file_type is invalid.');
end

const = vbmeg_inst.const;

% Analyze information for coordinate change
Vdim  = vb_analyze_info_get_voxdim(analyze_info);
Vsize = vb_analyze_info_get_voxsize(analyze_info);

for k=1:length(V_cel)
    % Convert coordinate from SPM_Right_m to analyze_right_mm
    V = vb_spm_right_to_analyze_right_mm(V_cel{k}, Vdim, Vsize);

    % make new surface data
    brain_data = surf_data_new(V, F_cel{k}, xx_cel{k}, ...
                                const.COORDINATE_ANALYZE_RIGHT_MM, ...
                                name_cel{k}, ...
                                conduct_cel{k});
    % set history
    comment = ['Imported from ' model_file];
    brain_data = brain_data_add_history_item(brain_data, comment);

    % add brain_data to bm_list
    obj.bm_list = bm_list_add_model(obj.bm_list, brain_data);
end

%
% --- After check
%
if nargout == 0
    error('function caller should receive obj.');
end
