function  [J, err, B] = vb_dipole_current(V, bexp, pick, Qpick, Wsensor, V0, mode)
% Dipole current time cource for dipoles at fixed position
%  [J, err, B] = vb_dipole_current(V, bexp, pick, Qpick, Wsensor, V0, mode)
% --- Output
%  J   : dipole current [NV, 3 ,Time]
%  J(n ,k ,t) =  current at n-th dipole ,time 't' and k-th direction
% err  : Reconstruction error
% B(m,t) : reconstructed magnetic field at m-th channel and time 't' 
% --- INPUT
% V      : position of dipoles (NV x 3)
% V(n,:) : n-th dipole position (3-D coordinate)
%
% bexp(m,t)   : MEG-data   (Nsensor ,Time) 
%
%  pick(k, 1:3) : Sensor coil position  : , 
% Qpick(k, 1:3)	: Sensor coil direction : 
%
% Wsensor(m,n) = n-th coil weight for m-th sensor channel
%    basis(channel,dipole) = Wsensor * basis(coil,dipole)
% --- Optional input
% V0   : Center of spherical model (1 x 3)
% mode = 0: Biot-Savart      , L = 3: independent current direction
%      = 1: Sarvas           , L = 2: independent current direction
%      = 2: Magnetic dipole  , L = 3: independent current direction
%      = 3: EEG              , L = 3: independent current direction
%
% 2006-12-16 made by M.Sato
% 2007-4-3 made by M.Sato
% 2007-7-9 modified by M.Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if ~exist('V0','var')|isempty(V0), V0 = [0 0 0]; end;
if ~exist('mode','var'), mode = 0; end;

T  = size(bexp,2);
NV = length(V(:))/3;		% Number of dipole position
V  = reshape(V,[NV 3]);

switch	mode
case	0
	% Magnetic field by Biot-Savart
	L  = 3;
case	1
	% Magnetic field by Sarvas
	L  = 2;
case	2
	% Magnetic field by magnetic dipole
	L  = 3;
case	3
	% EEG 3-shell Sphere model
	L  = 3;
end

% L = 2 : 
%   xx(n,:)      = 1st orthogonal vectors to V(n,:)
%   xx(n + NV,:) = 2nd orthogonal vectors to V(n,:)
% L = 3 :
%	xx(n       ,:) =  [1 0 0]
%	xx(n + NV  ,:) =  [0 1 0]
%	xx(n + 2*NV,:) =  [0 0 1]

% xx : current direction
[V,xx] = vb_current_vector(V,[],L);

% G : Lead Field (NV*L, Nsensor)
G = vb_dipole_basis(V, xx, pick, Qpick, Wsensor, V0, mode);

GGinv = pinv( G*G' ); % (NV*L x NV*L) matrix inverse

% Diploe current
JJ = (GGinv * G) * bexp;	% (NV*L x T)

% reconstructed field
B  = G' * JJ;

JJ = reshape(JJ, [NV, L, T]);

if L == 2,
	J  = zeros(NV,3,T);
	ix = 1:NV;
	
	for k = 1:3
		J(:,k,:) = JJ(ix,1,:) .* repmat(xx(ix,k), [1, 1, T]) ...
		         + JJ(ix,2,:) .* repmat(xx(ix + NV,k), [1, 1, T]) ;
	end
else
	J = JJ;
end

if nargout == 1, return; end;

% Reconstruction error

% MEG covariance matrix
BB    = bexp*bexp';	% BB(Nsensor,Nsensor)
Bnorm = sum(diag(BB));

% Error = ( B - G'*J )^2 = B^2  - ( B'* G'* GGinv * G * B )
err  = 1 - sum(sum(G.*(GGinv*G*BB)))/Bnorm;

return

% --- DEBUG

err2  = sum(sum((bexp-B).^2))/Bnorm;
delta = err - err2
