function [Jact ,ix_act, Jbck, ix_bck, Var, JactInfo] = ...
    vbmeg_current_reconstruct_tr(proj_root, bayesfile, ix_area, ...
			      tsubsamp, jactdir, mode)
% Current reconstruction for each trial using Bayesian inverse filter
% --- Syntax
% [Jact ,ix_act, Jbck, ix_bck, Varact, JactInfo] ...
%  = vbmeg_current_reconstruct_tr(proj_root, bayesfile, ...
%                                ix_area, tsubsamp, jactdir)
%  This program output current files for each trial to specified directory
%  If trial average current is needed, use 'vbmeg_current_reconstruct'
%
% --- Input
% bayesfile : result file obtained by VB estimation
%
% --- Optional Input
% ix_area : Position index to calculate estimated current
%           If 'ix_area' is empty or not given, 
%              currents in the active region are calculated
% tsubsamp : Specify subsampled time index
%           If 'tsubsamp' is empty or not given, 
%              time subsampling is not done
% jactdir  : output directory of current files for each trial
%            relative path from proj_root
%            do not include '/' or '\' in the end of string
%            If 'jactdir' is not given,  jactdir = 'temp'
%
% ---Output
% Jact    : active current
% ix_act  : Vertex index corresponding to active current
%
% Jact(n,t,:) is the current at the vertex 'ix_act(n)' & the time 't'
% Jact(Nact,Nsample,Ntrial) 
%   Nact     : # of active region, 
%   Nsample  : # of time sample, 
%   Ntrial   : # of trials 
%
% Jbck     : background current
% ix_bck   : Vertex index corresponding to background current
%            same for 'Jact'
% Varact   : posterior variance of act_area
%   When active area is empty,
%   posterior variance for background area is calculated
% ----
% MEG data : B{Nsession}(Nsensors, Tsample, Ntrials)
%
% 2004-9-25 M. Sato
% 2004-12-5 M. Sato : Modified
% 2004-12-22 M. Sato : Modified
% 2005-1-5  M. Sato Modified
% 2005-1-15  M. Sato Modified
% 2005-08-19 O. Yamashita
% * Posterior variance is calculated simultaneously
% * Variables are renamed
% * Corresponding to new bayesfile format
% 2005-08-19 O. Yamashita ver.30b
% 2005-09-30 O. Yamashita
% * Minor bug fix when Lact >= 2 & Lact_var = 1
% 2006-08-31 M. Sato
% * Remove focal area from global area
% * VBMEG result & liedfield is loaded by vb_load_result
% * Inverse filter calculation is done in vb_invfilter_calc
% 2009-04-02 Taku Yoshioka
%  Parameter name changed within this code for readability
%  (just replacing 'resultfile' to bayesfile)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)


if ~exist('ix_area','var'),  ix_area = [];end;
if ~exist('tsubsamp','var'), tsubsamp = [];end;
if ~exist('jactdir','var'),  jactdir = 'temp'; end;
if ~exist('mode','var'),     mode = 0; end;


%%%%%% load result
if ~isempty(proj_root)
	jactdir_ab = fullfile(proj_root, jactdir);
else
	jactdir_ab = [jactdir];
end  

%check directory
if exist(jactdir_ab) ~= 0
    msg = [ 'jact_dir : ' jactdir_ab ' exists. Do you over write?'];
    btn = questdlg(msg, 'confirm', 'Yes', 'No', 'Yes');
    if strcmp(btn, 'No')
        error('processing aborted.');
        return;
    end
end

if mkdir(proj_root, jactdir) ~= 1,  error('mkdir failed.'); end

%%%%%% load result and MEG data
[B, Model, nGact, nGall, nGGall, Cov, Wact, Wbck, ...
    bsnorm, ix_act, ix_bck, vb_parm] ...
    = vb_load_result(proj_root, bayesfile, ix_area);

% B      : MEG data
% Model  : Estimated model
% nGact  : Normalized leadfield of focal window
% nGall  : Normalized leadfield of global window
% Cov    : Sensor noise covariance matrix
% bsnorm : Normalizing constant
% Wact   : Spatial smoothing matrix of focal window
% Wbck   : Spatial smoothing matrix of global window
% ix_act : Vertex index corresponding to active current
% ix_bck : Vertex index corresponding to background current

% # of vertex
Nvact	   = vb_parm.Nvact;	     % # of active vertex 
Njact_area = vb_parm.Njact_area;
Njall_area = vb_parm.Njall_area;

%% Constant
Nsession = vb_parm.Nsession;	% Number of sessions
Ntrials  = zeros(Nsession,1);
Tsample  = vb_parm.Tsample;	    % Number of total time sample
Twindow  = vb_parm.Twindow;

% Temporal subsampling index
if isempty(tsubsamp),
	tsubsamp = 1:Tsample;
end

% Temporal smoothing window weight
[Tweight ,Tindex, Nindex, Nsample] = ...
    vb_overlapped_timewindow_weight(Twindow, Tsample, tsubsamp, mode);

Nwindow = length(Nindex);   	% # of time window

% Nsample : # of subsampled data	

if mode == 1,fprintf('Non-overlapped concatenation mode\n'); end;

%% Posterior variance
if Njact_area > 0
	Var    = zeros(Njact_area,Nsample);
	Varact = zeros(Njact_area,Nsample);
elseif Njall_area > 0
	Var    = zeros(Njall_area,Nsample);
	Varact = zeros(Njall_area,Nsample);
end

%% Current
Jact_ave = zeros(Njact_area,Nsample);
Jbck_ave = zeros(Njall_area,Nsample);

%% Estimated current variance
if isfield(Model,'a')
	a_inv = Model.a;
else
	a_inv = zeros(Nvact,Nwindow);
end
if isfield(Model,'v')
	v_inv = Model.v;
else
	v_inv = zeros(Nwindow,1);
end

%% Inverse filter
KW  = cell(Nwindow,1);
KW0 = cell(Nwindow,1);

Ntrial_all = 0;

% Session loop 
for n=1:Nsession
	Ntry = size(B{n},3);
	Ntrials(n) = Ntry;
	
	% Following notation is used in comments
	% G0 : original leadfield matrix (non-smoothed)   : [] 
	% A :  variance of active current                 : [A_inv]
	% tau : variance of active current                : [v_inv]
	% Wa : spatial smoothing matrix of active current : [Wact]
	% Wb : spatial smoothing matrix of background current : [Wback]
	% Ga : leadfield of active current (smoothed) G0*Wa   : [G]
	% Gb : leadfield of background current (smoothed) G0*Wb : []
	
	% Lead field for each session
	G	 = nGact{n};   % Ga 
	GbW  = nGall{n};   % Gb*Wb' 
	GG   = nGGall{n};  % Gb*Gb'
	Covs = Cov{n};     % Sg
	Nch	 = size(G,1);
	
	%%%% Calculate Inverse filter for current reconstruction
	for j=1:Nwindow
		% Time window loop
		% Subsampling time index
		Nid = Nindex{j};	% index in the total subsampled data
		Nt  = length(Nid);
		
		if isempty(Nid), continue; end;

		[KWt, KW0t, var_tmp] = vb_invfilter_calc(a_inv(:,j), v_inv(j), ...
		                          G, GbW, GG, Covs, Wact, Wbck);
		KW{j}  = KWt;
		KW0{j} = KW0t;
		var_tmp = Model.sx(j) * var_tmp./(bsnorm^2); 
		Varact(:,Nid) = Varact(:,Nid) + var_tmp * Tweight{j};
	end % Timewindow loop END
	
	%%%% Current reconstruction
	for m=1:Ntry
		Jact = zeros(Njact_area,Nsample);
		Jbck = zeros(Njall_area,Nsample);
		
		for j=1:Nwindow
			% Subsampling time index
			Tid = Tindex{j};	% subsampled time index
			Nid = Nindex{j};	% index in the total subsampled data
			
			if isempty(Nid), continue; end;
			
			weight = repmat(Tweight{j}, [Nch 1]);
			
			Bt = B{n}(:,Tid,m) .* weight; % Time window smoothing
			
			Jact(:,Nid) = Jact(:,Nid) + (KW{j} * Bt)/bsnorm;
			Jbck(:,Nid) = Jbck(:,Nid) + (KW0{j}* Bt)/bsnorm;
		end
		
		% Trial average current
		Jact_ave   = Jact_ave + Jact;
		Jbck_ave   = Jbck_ave + Jbck;
		Ntrial_all = Ntrial_all + 1;
		
		fname = sprintf('data_s%04dt%04d',n,m);
		vb_save([jactdir_ab '/' fname], 'Jact','Jbck', ...
		         'Jbck','ix_act','ix_bck','Varact');
		fprintf('progress... session:%04d/%04d, trial:%04d/%04d',...
			      n,Nsession,m,Ntry);
		%fprintf('%s',repmat(char(sprintf('\b')),1,46));
		fprintf('\n');
	end; % Trial loop END
	
	Var = Var + Ntry./max(Varact,eps);
end; % Session loop END

JactInfo.Ntrial   = Ntrials;
JactInfo.Nsession = Nsession;
JactInfo.jactdir  = jactdir;
JactInfo.NJact    = Njact_area;
JactInfo.Tsample  = Tsample;

% Trial average current
Jact = Jact_ave/Ntrial_all;
Jbck = Jbck_ave/Ntrial_all;

%Var  = Var/Ntrial_all;
Var  = 1./Var;
