function [Zact_ave ,Jinfo, bayes_parm, vb_parm, MEGinfo] ...
          = vbmeg_current_calc_z(proj_root, curr_parm);
% Current reconstruction using Bayesian inverse filter
%
% --- Syntax
%  [Zact ,Jinfo, bayes_parm, vb_parm, MEGinfo] ...
%          = vbmeg_current_calc_z(proj_root, curr_parm);
% --- Input
% curr_parm.resultfile : result file obtained by VB estimation
%
% --- Optional Input
% curr_parm.trial_average = [ON] : average current over all sessions 
%               = OFF  : current for each session
%
% curr_parm.ix_area : Position index to calculate estimated current
%           If 'ix_area' is empty or not given, 
%              currents in the active region are calculated
% curr_parm.tsubsamp   : Specify subsampled time index
%           If 'tsubsamp' is empty or not given, 
%              time subsampling is not done
% curr_parm.overlap_mode 
%   = 0 : current is averaged over overlapped time window
%   = 1 : current is not averaged for overlapped window
%         current time series of each time windows 
%         are concatenated sequentially for spectral analysis
% --- If following field is given, these values are used 
%     instead of bayes_parm field in result file
% curr_parm.basisfile
% curr_parm.megfile  
% curr_parm.twin_meg 
% curr_parm.Tperiod  
% curr_parm.Tnext    
% ---Output
% Zact    : active current
%
% Zact(n,t,:) is the current at the vertex 'ix_act(n)' & the time 't'
% Zact(Nact,Nsample)          for trial_average = ON 
% Zact(Nact,Nsample,Ntrials)  for trial_average = OFF
%   Nact     : # of active region, 
%   Nsample  : # of time sample, 
%   Ntrial  : # of trials in all session]
% Jinfo.ix_act  : Vertex index corresponding to active current
%
% * Inverse filter calculation is done in vb_invfilter_calc
% 2006-09-03 M. Sato
% * Non-overlapped concatenation mode is added for spectral analysis
% 2008-10-29 M. Sato
% * Inverse filter calculation is done in vb_invfilter_calc
% VBfilt.sx = Observation noise variance (Nwindow,Ntask)
% VBfilt.Hj = Model entropy (Nwindow,Ntask)
% VBfilt.KW = VB inverse filter  (Njact,Nch,Nwindow,Ntask)
% VBfilt.SB_cov = posterior sensor covariance matrix (Nch,Nch,Nwindow,Ntask)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

MinExp = -50;

% multiple bayse file for multiple conditions

if ~isempty(proj_root)
	filterfile = [proj_root filesep curr_parm.filterfile];
else
	filterfile = curr_parm.filterfile;
end  

%%%%%% load VBMEG estimated result
load(filterfile, 'VBfilt','Jinfo','bayes_parm','vb_parm');

%%%%%% check parameter of 'curr_parm'
% Values of 'curr_parm' fields dominates over 
%   those of 'bayes_parm' in bayesfile
[bayes_parm, ix_area, trial_average, tsubsamp, overlap_mode, verbose] = ...
			check_arg(bayes_parm, curr_parm);
			
if ~isempty(proj_root);
  bayes_parm_abs = vb_parm_absolute_path(proj_root, bayes_parm);
else
  bayes_parm_abs = bayes_parm;
end

% current file directory
if ~isempty(proj_root)
	currfile = fullfile(proj_root, curr_parm.currfile);
else
	currfile = [curr_parm.currfile];
end  

curr_root  = fileparts(currfile);

% jactdir is relative path from current file
if isfield(curr_parm,'jactdir') && ~isempty(curr_parm.jactdir)
	jactdir    = curr_parm.jactdir;
	jactdir_ab = fullfile(curr_root, jactdir);
	
	%check directory
	if exist(jactdir_ab,'dir') ~= 0
	    msg = [ 'jact_dir : ' jactdir_ab ' exists. Do you over write?'];
	    btn = questdlg(msg, 'confirm', 'Yes', 'No', 'Yes');
	    if strcmp(btn, 'No')
	        error('processing aborted.');
	        return;
	    end
	else
		res = mkdir(curr_root, jactdir);
		if  res~= 1,  error('mkdir failed.'); end
	end
else
	jactdir_ab = [];
end

%%%%%% MEG data preparation
% B      : MEG data
[B, Ntrial, Nch, Tsample, Twindow] = ...
	vb_megdata_preparation(bayes_parm_abs);

MEGinfo = vb_load_meg_info(bayes_parm_abs.megfile{1});

Nsession = length(B);	 % Number of files

fprintf('Number of session: %d\n', Nsession);

%% Constant

% Temporal subsampling index
if isempty(tsubsamp),
	tsubsamp = 1:Tsample;
end

%%%%%% Temporal smoothing window weight
[Tweight ,Tindex, Nindex, Nsample] = ...
    vb_overlapped_timewindow_weight(Twindow, Tsample, tsubsamp, overlap_mode);

Nwindow = size(Twindow,1);   	% # of time window

sx = VBfilt.sx;
Hj = VBfilt.Hj ;
KW = VBfilt.KW ;
SB_cov = VBfilt.SB_cov ;

% # of active vertex
Njact = size(KW,1);
Ntask = size(KW,4);
Nch = size(SB_cov,1);

fprintf('Active vertex number = %d\n',Njact)
fprintf('Active sensor number = %d\n',Nch)

if overlap_mode == 1,fprintf('Non-overlapped concatenation mode\n'); end;

%%%%%% Initialization
Zact = zeros(Njact,Nsample);
Zact_ave = zeros(Njact,Nsample);
LP   = zeros(Ntask,1);

% Current Info
Jinfo.Tsample = Tsample;
Jinfo.tindex  = tsubsamp;
Jinfo.Nwindow = Nwindow;
Jinfo.Ntask   = Ntask;

% MEG time window index
Tstart  = bayes_parm.twin_meg(1);
Tend    = bayes_parm.twin_meg(2);
if isempty(tsubsamp)
	Jinfo.Tsample = Tstart:Tend;
else
	Jinfo.Tsample = tsubsamp + Tstart - 1;
end

%%%%%% Time window loop
% MEG for each trial	
filename = [];

Ntrial_all = 0;
%nall = 0;

for n=1:Nsession
	Ntry = Ntrial(n);
	
	if Ntry == 0, continue; end;

	Bs = B{n};
	
	for m=1:Ntry
		
		Zact = zeros(Njact,Nsample);
		Post = zeros(Ntask,Nwindow);
		
		for j=1:Nwindow
			% Subsampling time index
			Tid = Tindex{j};	% subsampled time index
			Nid = Nindex{j};	% index in the total subsampled data
			Nt  = length(Tid);
			
			if Nt == 0, continue; end;
	
			%%%% Current reconstruction
			[Zp ,post]= vb_current_calc(Bs(:,Tid,m), ...
			            KW(:,:,j,:),SB_cov(:,:,j,:),Hj(j,:),sx(j,:));
			
			%%%% Time window smoothing
			weight = Tweight{j};
			Zact(:,Nid) = Zact(:,Nid) ...
			            + vb_repmultiply(Zp , weight);
		    Post(:,j)  = post;
			
		end % Timewindow loop
		
		% Trial average current
		Zact_ave   = Zact_ave + Zact;
		Ntrial_all = Ntrial_all + 1;
		
		if ~isempty(jactdir_ab)
			fname = sprintf('data_s%04dt%04d',n,m);
			filename{m} = fname;
			vb_fsave([jactdir_ab '/' fname], 'Zact','Jinfo','Post');
			if verbose==1,
				fprintf('.')
                if rem(m, 20) == 0 % linefeed per 20
                    fprintf('\n');
                end
			end
		end
	end; % Trial loop
end

Zact_ave = Zact_ave/Ntrial_all;

Jinfo.Ntrial   = Ntrial;
Jinfo.Nsession = Nsession;
Jinfo.jactdir  = jactdir;
Jinfo.filename = filename;

if verbose==1,
	fprintf('\n')
end

% ix_act : Vertex index corresponding to active current Zact
% ix_act_ex : Vertex index corresponding to active current Jact
% Wact   : Spatial smoothing matrix of focal window
% Jact   = Wact * Zact

return

%%%% ---------------
function [bayes_parm,ix_area,trial_average,tsubsamp,overlap_mode,verbose]= ...
			check_arg(bayes_parm,curr_parm)

if isfield(curr_parm,'basisfile'), 
	bayes_parm.basisfile = curr_parm.basisfile;
end;
if isfield(curr_parm,'megfile'), 
	bayes_parm.megfile   = curr_parm.megfile  ;
end;
if isfield(curr_parm,'twin_meg'), 
	bayes_parm.twin_meg  = curr_parm.twin_meg ;
end;
if isfield(curr_parm,'Tperiod'), 
	bayes_parm.Tperiod   = curr_parm.Tperiod  ;
end;
if isfield(curr_parm,'Tnext'), 
	bayes_parm.Tnext     = curr_parm.Tnext    ;
end;

if ~isfield(curr_parm,'trial_average'), 
	trial_average = ON; 
else
	trial_average = curr_parm.trial_average; 
end;

bayes_parm.trial_average = trial_average;

if ~isfield(curr_parm,'ix_area'),  
	ix_area = []; 
else
	ix_area = curr_parm.ix_area; 
end;
if ~isfield(curr_parm,'tsubsamp'), 
	tsubsamp = []; 
else
	tsubsamp = curr_parm.tsubsamp; 
end;
if ~isfield(curr_parm,'overlap_mode'),     
	overlap_mode = 0; 
else
	overlap_mode = curr_parm.overlap_mode; 
end;
if ~isfield(curr_parm,'verbose'),     
	verbose = 1; 
else
	verbose = curr_parm.verbose; 
end;
