function [CW ,ixall] = vb_spatial_gauss_filter(brainfile,Rradius,Rmax,Index,flag,cosval,normalize,smoother)
% Calculate gasussian smoothing filter of 2D cortical surface 
% --- Syntax
% function [CW ,ixall] =
% vb_spatial_gauss_filter(brainfile,Rradius,Rmax,Index,flag,cosval,normalize)
%
% --- Input
% brainfile   : brain file 
% Rradius     : Gaussian smoothing filter radius
% Rmax        : Max radius of Gaussian smoothing filter
% Index       : Vertex index for selected area
% --- Optional Input
% flag = 0    : output region = input region
%      = 1    : extend output region to neighbor of edge (Default)
% cosval : Threshold value for normal direction cos = cos(limit_angle) [-1]
% normalize = 0: Normalization is not applied. 
% normalize = 1: Normalization of spatial filter coefficient is applied
%                (Default). 
% smoother    : smoothing fiter is made from
%              'std' : standard brain (nextIX, nextDD, xx).  (Default)
%                      If the brainfile is V1 format, Individual brain is used.
%              'subj': Individual brain (subj.nextIX, nextDD, xx).
% --- Output
% CW          : Smoothing Gaussian filter
% ixall       : Vertex index for smoothing filter
%               smoothed_leadfield(:,Index) = leadfield(:,ixall) * CW
%
% Ver 1.0 written by M. Sato  2003-4-15
% Modified by Taku Yoshioka
% 2005-3-29 O.Yamashita
%   Bug fix : the program does not run due to 'Index' 
%             after reudce_neighbor_data, when flag = 0. 
%   Error check for parm.cosval.
%   The comment is modified.  
% 2005-3-31 O.Yamashita
%   Bug fix : 'ixall' is not output correctly, when flag = 0. 
% 2005-08-09 O.Yamashita
%   input arguments are modified
% 2010-12-14 taku-y
%  [minor] Input argument 'normalize' added. 
% 2017-03-15 rhayashi
%  [enhance] coord_type is added.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Set filter radius
%Rmax	= Rradius * 2;

% Default parameters
if ~exist('flag', 'var')       || isempty(flag),      flag = 1; end
if ~exist('cosval', 'var')     || isempty(cosval),    cosval = -1; end
if ~exist('normalize', 'var')  || isempty(normalize), normalize = 1; end
if ~exist('smoother', 'var')   || isempty(smoother),  smoother = 'std'; end

% smoothing information
switch(lower(smoother))
    case 'std'
        [nextDD, nextIX] = vb_load_cortex_neighbour(brainfile);
        [tmp1, tmp2, xx] = vb_load_cortex(brainfile);
    case 'subj'
        [nextDD, nextIX] = vb_load_cortex_neighbour(brainfile, 'subj');
        [tmp1, tmp2, xx] = vb_load_cortex(brainfile, 'subj');
    otherwise
        error('Unknown coord_type is specified.');
end

Nvertex = length(Index);	% # of points in selected area
Nall	= length(nextIX);	% # of points in whole brain


if flag == 1 
    rIndex = Index;   
else
    % Restrict neighbor vertex into input index region
	[nextIX,nextDD] = ...
    	vb_reduce_neighbor_data(Index,nextIX,nextDD);
    rIndex = [1:Nvertex]'; %rename index
end;

% Threshold value for normal direction cos
if cosval > 1 % avoid empty index in for loop
    cosval = -1;
end

%%%%%%% Smoothing Filter calculation %%%%%%%
if	Rradius <= 0,
	% No smoothing 
	CW    = speye(Nvertex,Nvertex);
	ixall = Index;  
	return
end

% Calculate total number of neighbors (Nnext)
Nnext   = 0;

for n=1:Nvertex,
  inx   = find( nextDD{rIndex(n)} <= Rmax );	% neighbor within Rmax
  Nnext = Nnext + length(inx);
end;

% Initialize variable
val 	= zeros(Nnext,1);	% Filter value
ipoint	= zeros(Nnext,1);	% Center point index
inext	= zeros(Nnext,1);	% Neighbor index
Nbegin  = 1;

% Find neighbor within Rmax and calculate Gaussian filter value
for n=1:Nvertex,
	i	= rIndex(n);				% center vertex index
	dd0 = nextDD{i};			% Neighbor distance
	inx = find( dd0 <= Rmax );	% Find neighbor within Rmax
	
	ix	= nextIX{i}(inx);		% Neighbor index within Rmax
	dd0 = dd0(inx);				% Neighbor distance
	
	% Check direction cos is larger than cosval
	dcos= xx(ix,:)*xx(i,:)';
	inx = find( dcos > cosval );
	ix  = ix(inx);
	
	% Gaussian filter value
	vdd = exp( - (dd0(inx)/Rradius).^2 );	% Gaussian filter
        if normalize, 
          vdd = vdd/sum(vdd);			% Normalization
        end
	
	% Total # of neighbor points
	Nend   = Nbegin + length(inx) - 1;
	Indx   = Nbegin:Nend;
	Nbegin = Nend + 1;
	
	ipoint(Indx) = n;		% center vertex index
	inext(Indx)  = ix ;     % Neighbor index
	val(Indx)    = vdd;		% Gaussian filter value

end;

ipoint = ipoint(1:Nend);
inext  = inext(1:Nend) ;
val    = val(1:Nend)   ;
    
CW    = sparse( inext , ipoint , val , Nall, Nvertex) ;
ixall = unique(inext);
CW    = CW(ixall,:);        
if flag == 0, ixall = Index; end % index before renamed.




