function [B,Ntrial,Nsensor,Tsample,Tmsec] ...
    = vb_get_megdata(megfile,twin_meg,temporal_filter,trial_average,ix_trial)
% Get megdata for VB estimate;
%
% [B, Ntrial, Nsensor, Tsample] = vb_get_megdata(megfile, twin_meg,
% temporal_filter, trial_average)
%
% B{Nsession}   : MEG data concatenating all sessions
%                 Sensor number/position for each session may be different
%               : B{Nsession}(Nsensor, Tsample, Ntrial)
%
% Tsample : Total time sample of each trial for the analysis
% Ntrial  : # of trials in each session
% Nsensor : # of MEG sensors in each session 
%
% [history]
% 2004-11-16 Taku Yoshioka
% Modified by M.Sato 2004/12/22
% rearranged by O. Yamashita
% 2008-10-14 Taku Yoshioka
% 2008-11-28 Taku Yoshioka
%   Use vb_disp() for displaying message
% 2011-05-11 taku-y
%  [minor] New output variable 'Tmsec' was added. This is a vector of
%  actual time of row of B{n}. 
% 2012-01-27 taku-y
%  [minor] Messages have been modified.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

const = vb_define_verbose;
VERBOSE_LEVEL_NOTICE = const.VERBOSE_LEVEL_NOTICE;
VERBOSE_LEVEL_EMERGENCY = const.VERBOSE_LEVEL_EMERGENCY;

if nargin<5, ix_trial = []; end
if nargin<4 | isempty(trial_average), trial_average = false; end
if nargin<3 | isempty(temporal_filter), temporal_filter = false; end

% MEG data file names given as a cell array
if iscell(megfile),
  Nfile = length(megfile);
else
  tmp = megfile;
  megfile = {tmp};
  Nfile = 1;
end

MEGinfo = vb_load_meg_info(megfile{1});
Tstart = vb_index_to_time(twin_meg(1), MEGinfo);
Tend   = vb_index_to_time(twin_meg(2), MEGinfo);
vb_disp(['  Number of sessions          : ' num2str(Nfile)], ...
        VERBOSE_LEVEL_NOTICE);
vb_disp(['  Time window                 : ' num2str(Tstart,'%4.1f') ...
         ' - ' num2str(Tend,'%4.1f') ' [ms]'],VERBOSE_LEVEL_NOTICE);

% Time window for analysis
Tfilt	= twin_meg(1):twin_meg(2);
% Total sample for each trial
Tsample = twin_meg(2) - twin_meg(1) + 1;

B       = cell(Nfile,1); % MEG data 
Ntrial  = zeros(Nfile,1);
Nsensor = zeros(Nfile,1);

for n = 1:Nfile
  % Load MEG data
  bexp = vb_load_meg_data(megfile{n});
  %  load(megfile{n},'bexp');
  if ~isempty(ix_trial), bexp = bexp(:,:,ix_trial); end
  
  % MEG trial averaging
  if trial_average, 
    bexp = mean(bexp,3); 
  end
  
  % Variable preparation
  Ntrial(n)  = size(bexp,3);
  Ndata      = size(bexp,2);
  Nsensor(n) = size(bexp,1);
  
  if Ndata < twin_meg(2), 
    vb_disp(['Error: Load MEG file ' megfile{n}], ...
            VERBOSE_LEVEL_EMERGENCY);
    error(['# of sample is smaller than given period' ])
  end

  vb_disp(sprintf('  MEG data file (session %2d)  : %s',n,megfile{n}), ...
          VERBOSE_LEVEL_NOTICE);
  vb_disp(['  Number of sensors           : ' num2str(Nsensor(n))], ...
          VERBOSE_LEVEL_NOTICE);
  vb_disp(['  Number of trials            : ' num2str(Ntrial(n))], ...
          VERBOSE_LEVEL_NOTICE);

  % MEG data
  Bn = bexp(:,Tfilt,:);

  % MEG temporal smoothing 
  if temporal_filter
    for s = 1:Ntrial(n)
      Bn(:,:,s) = vb_temporal_smooth( Bn(:,:,s) );
    end
  end

  B{n} = Bn;
end

% actual time (all sessions assumed to have the same length, pretrigger)
MEGinfo = vb_load_meg_info(megfile{1});
Tmsec   = vb_index_to_time(Tfilt, MEGinfo);

return;