function [trigger_channel_idx, trigger_sample_idx] = ...
  vb_msrmnt_auto_detect_trigger_info(msrmnt_file, ch_type)
% get trigger information from msrmnt_file by auto-detect
%
% [usage]
%   [trigger_channel_idx, trigger_sample_idx] = ...
%     vb_msrmnt_auto_detect_trigger_info(msrmnt_file, ch_type)
%
% [input]
%   msrmnt_file : <required> actually EEG-MAT(.eeg.mat) file
%       ch_type : <optional> 1 or 2 [1]
%               : channel type
%               : 1) standard
%               : 2) hardware
%
% [output]
%   trigger_channel_idx : index of channels of trigger (cell array)
%                       : normally one
%    trigger_sample_idx : <struct> sample numbers information of trigger
%                      : [1 x N_channel]
%            :  - all_idx : list of index of "ON" bit
%            :  - beg_idx : index that "ON" bit begins
%            :  - end_idx : index that "ON" bit breathe
%
% [note]
%
% [history]
%   2007-01-10 (Sako) initial version
%   2011-06-01 (Sako) converted return values of vb_load_device to upper case
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

vb_define_device;

% --- CHECK ARGUMENTS --- %
if ~exist('msrmnt_file', 'var'), msrmnt_file = []; end;
if ~exist('ch_type', 'var'), ch_type = []; end;

[msrmnt_file, ch_type, Measurement, info, data] = ...
  inner_check_arguments(msrmnt_file, ch_type);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
% auto-detect trigger channel
func_ = mfilename;

switch Measurement
  case  'EEG'
    % return values
    trigger_channel_idx = [];
    trigger_sample_idx = [];

    [idx, trigger_channel] = vb_eeginfo_get_trigger_status_label(info);

    status_data = data(idx,:);

    [ch_list, bit_list] = ...
      vb_eeg_get_active_trigger_channel(status_data, DEFAULT_BIT_LEN);

    valid_ch_idx = find(ch_list <= DEFAULT_TRIGGER_CH_NUM);
    valid_bit_list = bit_list(valid_ch_idx,:);
    
    trigger_channel_idx = valid_ch_idx;

    trigger_sample_idx = [];
    for nch = 1:size(valid_bit_list,1)
      trigger_sample_idx = [trigger_sample_idx; ...
        vb_util_get_bit_on_list(valid_bit_list(nch,:), DEFAULT_BIT_LEN)];
    end
      
  case  'MEG'
    error('(%s)not be implemented yet for ''%s''', func_, Measurement);
    
  otherwise
    error('(%s)unknown Measurement: ''%s''', func_, Measurement);
end

% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [msrmnt_file, ch_type, Measurement, info, data] = ...
  inner_check_arguments(msrmnt_file, ch_type)
func_ = mfilename;

if isempty(msrmnt_file)
  error('(%s)msrmnt_file is a required parameter', func_);
end

if exist(msrmnt_file, 'file') ~= 2
  error('(%s)cannot find measurement file : %s', func_, msrmnt_file);
end

if isempty(ch_type)
  ch_type = 1;  % standard
end
    
[measurement] = vb_load_device(msrmnt_file);
Measurement = upper(measurement);

switch Measurement
  case  'EEG'
    load(msrmnt_file, 'EEGinfo', 'eeg_data');
    if ~exist('EEGinfo', 'var') || isempty(EEGinfo)
      error('(%s)measurement file doesnot have EEGinfo field', func_);
    end

    if ~exist('eeg_data', 'var') || isempty(eeg_data)
      error('(%s)measurement file doesnot have eeg_data field', func_);
    end
    info = EEGinfo;
    data = eeg_data;
    
  case  'MEG'
    error('(%s)not be implemented yet for ''%s''', func_, Measurement);
    
  otherwise
    error('(%s)unknown Measurement: ''%s''', func_, Measurement);
end
%
% --- end of inner_check_arguments()

% --- END OF FILE --- %
