function	[y0, z0, z] = vb_get_emg_threshold(y,parm)
% Estimate threshold for smoothed EMG power
%    [y0, z0, z] = vb_get_emg_threshold(y,parm)
%    
% --- Input
% y             : EMG signal
% 
% parm.fsamp    : sampling frequency [Hz]
% --- Optional parameter for EMG onset
% parm.p_val : P-value corresponding to the threshold   [0.001]
% parm.t_smooth : moving average window length          [20 ms]
%
% --- Output
% y0 = z0 : threshold value estimated from moving average
% z  : smoothed EMG power by TKE operator
%
% 2009-09-05 Masa-aki Sato
% 2012-1-8   Masa-aki Sato  set z0 = y0
% 2012-2-18  Masa-aki Sato  use TKE operator
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Default value
mode = 1;  % low-pass filtering 
t_smooth  = 20;  % 10/20 ms average (100/50 Hz)
p_val = [0.001];

if isfield(parm,'t_smooth'), t_smooth = parm.t_smooth;end
if isfield(parm,'p_val'),    p_val  = parm.p_val ;end

% sampling frequency
fs  = parm.fsamp ; 

switch	mode
case	0
	% smoothed signal for threshold estimation
	tau  = round(t_smooth * fs/ 1000);  % msec -> sample number
	yave = filter( ones(tau,1)/tau, 1, abs(y));
case	1
	% low-pass filtering 
	Norder = 2;
	[B,A] = butter(Norder, 1000/(t_smooth * fs /2) ,'low');
	yave  = filtfilt(B,A, abs(y) );
end

% Estimate threshold by gamma distribution approximation
[y0, A ,hy ,ylist] = vb_gamma_dist_param(yave, p_val(1));

% TKE operator EMG filter
z = vb_emg_filter(y, parm.fsamp);

% set (smoothed EMG threshold) = (EMG threshold)
z0 = y0;

if isfield(parm,'plot_mode') && parm.plot_mode > 0
	vb_show_hist_threshold(y0, A ,hy ,ylist);
	Nbin = 1000;
	[hz, zx] = hist(z,Nbin);
	figure;
	plot(zx,hz);
	hold on
	plot([z0 z0],[0 max(hz)*0.2],'r-')
	xlim([0 z0*3]);
end

return
% ---- END of program ----
