function [pick, Qpick, Nref] = ...
  vb_meg_yokogawa_load_ref_sensor(channel_info_base)
% load reference sensor coordinates from YOKOGAWA file
%
% [usage]
%   [pick, Qpick, Nref] = vb_meg_yokogawa_load_ref_sensor(channel_info_base)
%
% [input]
%   channel_info_base : <required> <<struct>> return value of getYkgwHdrChannel
%
% [output]
%    pick : [Nch x 3] sensor coordinates
%   Qpick : [Nch x 3] vector
%    Nref : number of reference channel
%
% [note]
%   
% [history]
%   2008-03-28 (Sako) thoroughly modified
%   2011-06-22 (Sako) Completely renewed to be adapted to the new library
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('channel_info_base', 'var'), channel_info_base = []; end
[channel_info_base] = inner_check_arguments(channel_info_base);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
% --- <<struct array>> fields are 'type' and 'data'
ch_info_list = channel_info_base.channel;

% '4' means reference channel
ref_ch_types = vb_yokogawa_get_channel_type(4);

Channel_active = [];
Channel_name = [];
Channel_type = [];
Channel_id = [];

% --- channel information list [1 x n_channel]
ch_type_list = [ch_info_list(:).type];

for i_type = 1:length(ref_ch_types)
  cur_type = ref_ch_types(i_type).id;

  cur_idx = find(ch_type_list(:) == cur_type);
  
  if isempty(cur_idx)
    continue;
  end

  num_ch = length(cur_idx);
  

  % active or not (NullChannel or not)
  Channel_active = [Channel_active; ones(num_ch,1)];
  Channel_name   = [Channel_name; (cur_idx + 1)];
  Channel_type   = [Channel_type; ch_type_list(cur_idx)];
  Channel_id     = [Channel_id; cur_idx];
end

Nref = length(Channel_active == 1);

if Nref == 0
  pick = [];
  Qpick = [];
  return;
end

%
% -- Coil information 
%
pick = zeros(Nref,3);
normline= zeros(Nref,3);
unit_vector= zeros(Nref,3);

% --- current (reference channel) list
cur_data_list = [ch_info_list(Channel_id).data];

% --- NewYKGWLib field format of channel_info.channel.data ------------------- %
%
% ----- If channel type is AxialGradioMeter or ReferenceAxialGradioMeter
%
% .x          : x coordinate of inner coil position [meter]
% .y          : y coordinate of inner coil position [meter]
% .z          : z coordinate of inner coil position [meter]
% .zdir       : Sensor orientation from z-axis [degree]
% .xdir       : Sensor orientation from x-axis [degree]
% .baseline   : Baseline length [meter]
% .size       : Inner coil size [meter]
% .name       : Abbreviation name
%
% ---------------------------------------------------------------------------- %

pick(:, 1) = [cur_data_list(:).x]; % X
pick(:, 2) = [cur_data_list(:).y]; % Y
pick(:, 3) = [cur_data_list(:).z]; % Z

% normal line of pick
zdir = [cur_data_list(:).zdir] * (pi/180); % zdir1 -- 6
xdir = [cur_data_list(:).xdir] * (pi/180); % xdir1 -- NOTICE -- not 8

normline(:, 1) = (sin(zdir) .* cos(xdir));
normline(:, 2) = (sin(zdir) .* sin(xdir));
normline(:, 3) = (cos(zdir));
size_normline = ...
  sqrt(normline(:, 1).^2 + normline(:, 2).^2 + normline(:, 3).^2);
unit_vector(:, 1) = normline(:, 1) ./ size_normline(:);
unit_vector(:, 2) = normline(:, 2) ./ size_normline(:);
unit_vector(:, 3) = normline(:, 3) ./ size_normline(:);

%
% normal line vector of pick
%
Qpick = unit_vector;

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [channel_info] = inner_check_arguments(channel_info)
func_ = mfilename;
if isempty(channel_info)
  error('(%s) channel_info is a required parameter', func_);
end

% --- check new channel_info
if ~isfield(channel_info, 'channel_count') ...
    || ~isfield(channel_info, 'channel')
  error('(%s) channel_info is not what is expected', func_);
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%--- END OF FILE ---%
