function vb_yokogawa_plot_sensor(megfile, plot_spec)
% Plot yokogawa sensor.
% [USAGE]
%    vb_yokogawa_plot_sensor(<megfile> [,plot_spec]);
% [IN]
%    megfile : <<required>> MEG-MAT file.
%    plot_spec <<optional>> struct of plot parameter
%             .AxisBackgroundColor : Background color of axes
%                                    = 'k' %  [DEFAULT]
%             .ChannelType   : plot channel type
%                              = 0  % all     [DEFAULT]
%                              = 1  % Axial  vb_gradio meter
%                              = 2  % Planar vb_gradio meter
%             .PlotDetectorCoil : plot detector coil
%                                  = true   % plot on  [DEFAULT]
%                                  = false  % plot off
%                .DetectorMtype : plot type of detector coil.
%                                 = 'r.'   [DEFAULT]
%                .DetectorMsize : plot size of detector coil.
%                                  = 30    [DEFAULT]
%             .PlotCompensationCoil : plot compensation coil
%                                      = true   % plot on
%                                      = false  % plot off [DEFAULT]
%                .CompensationMtype : plot type of compensation coil.
%                                      = 'c.'   [DEFAULT]
%                .CompensationMsize : plot size of compensation coil.
%                                      = 30    [DEFAULT]
%             .ConnectPairLine : connect pair of coils with line
%                                  = 'b-'   [DEFAULT]
%             .PlotNormalVector : plot normal vector
%                                  = true   % plot on  [DEFAULT]
%                                  = false  % plot off
%                 .VectorLength : normal vector length[m]
%                                  = 0.01   [DEFAULT]
%             .PlotSphere : plot sphere
%                            = true   % plot on  [DEFAULT]
%                            = false  % plot off
%               .SphereRadius : Sphere radius [m]
%                                = 0.1           [DEFAULT]
%                .SphereColor : Sphere color
%                                = [0.8 0.8 0.8] [DEFAULT]
% [OUT]
%    none
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%

if ~exist('megfile', 'var') || exist(megfile, 'file') ~= 2
    error('megfile is a required parameter.');
end

parm = DEFAULT_PARAMETER;

% Check inputted fields
if exist('plot_spec', 'var')
    fields = fieldnames(plot_spec);
    Nfields = length(fields)
    for k=1:Nfields
        if ~isfield(parm, fields{k})
            error('Unknown parameter : %s', fields{k});
        end
    end

    % Set parameter to parm
    for k=1:Nfields
        parm.(fields{k}) = plot_spec.(fields{k});
    end
end

%
% --- Main Procedure
%

% constant
MagnetoMeter          = 1;
AxialGradioMeter      = 2;
PlanarGradioMeter     = 3;

% data load
load(megfile, 'pick', 'Qpick', 'MEGinfo');
CoilWeight = MEGinfo.sensor_weight;
if isfield(MEGinfo, 'ChannelInfo')
    ChannelInfo = MEGinfo.ChannelInfo;
else
    ChannelInfo.Type = repmat(AxialGradioMeter, [MEGinfo.Nch, 1]);
    ChannelInfo.Name = MEGinfo.MEGch_name;
end

% --- AxialGradioMeter
ix_axial = find(ChannelInfo.Type == AxialGradioMeter  );
ChannelName = ChannelInfo.Name;
Nch_axial = length(ix_axial);
ix_axial_pair = zeros(Nch_axial, 1);
for n=1:Nch_axial
    ix_axial_pair(n) = find(CoilWeight(ix_axial(n),:) < 0 );
end
pick1_axial = pick(ix_axial,:);
pick2_axial = pick(ix_axial_pair,:);
Qpick1_axial = Qpick(ix_axial,:);
Qpick2_axial = Qpick(ix_axial_pair,:);

% --- PlannerGradioMeter
ix_plane = find(ChannelInfo.Type == PlanarGradioMeter);
Nch_plane = length(ix_plane);
ix_plane_pair = zeros(Nch_plane, 1);
for n=1:Nch_plane
    ix_plane_pair(n) = find(CoilWeight(ix_plane(n),:) < 0 );
end
pick1_plane = pick(ix_plane,:);
pick2_plane = pick(ix_plane_pair,:);
Qpick1_plane = Qpick(ix_plane,:);
Qpick2_plane = Qpick(ix_plane_pair,:);


%%%%%%%%%%%%%%%%%%%%%%
%   Plot section
%%%%%%%%%%%%%%%%%%%%%%

fig = figure;
h = axes; hold on;
set(h, 'Color', parm.AxisBackgroundColor);

% Detector coil
if parm.PlotDetectorCoil
    if( parm.ChannelType == 0 || parm.ChannelType == 1 )
        % Axial Detector coil
        plot3(pick1_axial(:, 1), pick1_axial(:, 2), pick1_axial(:, 3), ...
              parm.DetectorMtype, ...
              'MarkerSize', parm.DetectorMsize);
        % Axial Detector coil normal vector
        if parm.PlotNormalVector
            dx = parm.VectorLength;
            pos_x = [pick1_axial(:, 1), pick1_axial(:, 1) + dx * Qpick1_axial(:, 1)]';
            pos_y = [pick1_axial(:, 2), pick1_axial(:, 2) + dx * Qpick1_axial(:, 2)]';
            pos_z = [pick1_axial(:, 3), pick1_axial(:, 3) + dx * Qpick1_axial(:, 3)]';
            plot3(pos_x, pos_y, pos_z, '-');
        end
    end
    if( parm.ChannelType == 0 || parm.ChannelType == 2 )
        % Planar Detector coil
        plot3(pick1_plane(:, 1), pick1_plane(:, 2), pick1_plane(:, 3), ...
              parm.DetectorMtype, ...
              'MarkerSize', parm.DetectorMsize);
        % Planar Detector coil normal vector
        if parm.PlotNormalVector
            dx = parm.VectorLength;
            pos_x = [pick1_plane(:, 1), pick1_plane(:, 1) + dx * Qpick1_plane(:, 1)]';
            pos_y = [pick1_plane(:, 2), pick1_plane(:, 2) + dx * Qpick1_plane(:, 2)]';
            pos_z = [pick1_plane(:, 3), pick1_plane(:, 3) + dx * Qpick1_plane(:, 3)]';
            plot3(pos_x, pos_y, pos_z, '-');
        end
    end
end

% Compensation Coil
if parm.PlotCompensationCoil

    if( parm.ChannelType == 0 || parm.ChannelType == 1 )
        % Axial Compensation Coil
        plot3(pick2_axial(:, 1), pick2_axial(:, 2), pick2_axial(:, 3), ...
            parm.CompensationMtype, ...
            'MarkerSize', parm.DetectorMsize);
        % Axial Compensation Coil normal vector
        if parm.PlotNormalVector
            dx = parm.VectorLength;
            pos_x = [pick2_axial(:, 1), pick2_axial(:, 1) + dx * Qpick2_axial(:, 1)]';
            pos_y = [pick2_axial(:, 2), pick2_axial(:, 2) + dx * Qpick2_axial(:, 2)]';
            pos_z = [pick2_axial(:, 3), pick2_axial(:, 3) + dx * Qpick2_axial(:, 3)]';
            plot3(pos_x, pos_y, pos_z, '-');
        end
    end
    if( parm.ChannelType == 0 || parm.ChannelType == 2 )
        % Planar Compensation coil
        plot3(pick2_plane(:, 1), pick2_plane(:, 2), pick2_plane(:, 3), ...
              parm.CompensationMtype, ...
              'MarkerSize', parm.DetectorMsize);
        % Planar Compensation coil normal vector
        if parm.PlotNormalVector
            dx = parm.VectorLength;
            pos_x = [pick2_plane(:, 1), pick2_plane(:, 1) + dx * Qpick2_plane(:, 1)]';
            pos_y = [pick2_plane(:, 2), pick2_plane(:, 2) + dx * Qpick2_plane(:, 2)]';
            pos_z = [pick2_plane(:, 3), pick2_plane(:, 3) + dx * Qpick2_plane(:, 3)]';
            plot3(pos_x, pos_y, pos_z, '-');
        end
    end
end

% Connect detector coil and compensation coil.
if parm.PlotPairLine
    if parm.PlotDetectorCoil && parm.PlotCompensationCoil
        if( parm.ChannelType == 0 || parm.ChannelType == 1 )
            % Axial pair
            pos_x = [pick1_axial(:, 1), pick2_axial(:, 1)]';
            pos_y = [pick1_axial(:, 2), pick2_axial(:, 2)]';
            pos_z = [pick1_axial(:, 3), pick2_axial(:, 3)]';
            plot3(pos_x, pos_y, pos_z, parm.PairLineType);
        end
        if( parm.ChannelType == 0 || parm.ChannelType == 2 )
            % Planar pair
            pos_x = [pick1_plane(:, 1), pick2_plane(:, 1)]';
            pos_y = [pick1_plane(:, 2), pick2_plane(:, 2)]';
            pos_z = [pick1_plane(:, 3), pick2_plane(:, 3)]';
            plot3(pos_x, pos_y, pos_z, parm.PairLineType);
        end
    end
end

% Sphere
if parm.PlotSphere
    [F, V] = vb_make_fullsphere(1000);
    V = V * parm.SphereRadius;
    patch('Faces', F, 'Vertices', V, 'Edgecolor', 'none', ...
          'FaceColor', parm.SphereColor);
end

axis equal;
axis tight;

% Figure title
Title = [];
if parm.PlotDetectorCoil
    Title = 'Detector Coils';
end
if parm.PlotCompensationCoil
    if ~isempty(Title)
        Title = [Title, ' & '];
    end
    Title = [Title, 'Compensation Coils'];
end

Title = [Title, '( Axial gradiometer and Planar gradiometer )'];

if parm.ChannelType == 2 || isempty(pick1_axial)
   Title = strrep(Title, 'Axial gradiometer', '');
   Title = strrep(Title, 'and ', '');
end
if parm.ChannelType == 1 || isempty(pick1_plane)
   Title = strrep(Title, 'Planar gradiometer', '');
   Title = strrep(Title, 'and ', '');
end
   
set(fig, 'Name', Title);
set(fig, 'NumberTitle', 'off');

rotate3d on;

return;
%
% --- Inner function
%

function plot_spec = DEFAULT_PARAMETER
% set default parameter of this function.
%
plot_spec = struct;

% Plot channel type
plot_spec.AxisBackgroundColor = 'k';
plot_spec.ChannelType = 0;

% Detector Coil
plot_spec.PlotDetectorCoil = true;
plot_spec.DetectorMtype = 'r.';
plot_spec.DetectorMsize = 30;

% Compensation Coil
plot_spec.PlotCompensationCoil  = true;
plot_spec.CompensationMtype = 'c.';
plot_spec.CompensationMsize = 30;

% ConnectLine between detector coil and Compensation coil.
plot_spec.PlotPairLine = false;
plot_spec.PairLineType = 'b-';

% NormalVetorLength [m]
plot_spec.PlotNormalVector = true;
plot_spec.VectorLength     = 0.01;

% Sphere plot
plot_spec.PlotSphere = true;
plot_spec.SphereRadius = 0.1;
plot_spec.SphereColor  = [0.8 0.8 0.8];

