function [Nchannel, result] = ...
  vb_meginfo_get_channel_number_extra(meginfo, ex_type, active_swt)
% get number of extra channel from MEGinfo struct
% [usage]
%   [Nchannel, result] = ...
%      vb_meginfo_get_channel_number_extra(meginfo, ex_type, active_swt)
% [input]
%      meginfo : <required> <<struct>> MEG information
%      ex_type : <optional> extra type 0 | 10
%              :  [0]) except reference magnetometer channels
%              :  10 ) include reference magnetometer channels (as old)
%   active_swt : <optional> <<boolean>> [false] switch for active filter
%              :   true) only active channels
%              :  false) all the channels
% [output]
%   Nchannel : number of extra channel
%     result : <<integer>> error code
%            :  0) success
%            :  1) error - bad MEGinfo
%            :  2) error - bad parameter (ex_type)
% [note]
%   in new specification, refference magnetometer channel (refmg_ch) is 
%   set apart from extra channels
%   only 'YOKOGAWA'
% [history]
%   2007-06-27 (Sako) initial version
%   2008-02-01 (Sako) modified according to new channel_label spec
%   2008-04-21 (Sako) added active_swt
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('meginfo', 'var'), meginfo = []; end
if ~exist('ex_type', 'var'), ex_type = []; end
if ~exist('active_swt', 'var'), active_swt = []; end
[meginfo, ex_type, active_swt, result] = ...
  inner_check_arguments(meginfo, ex_type, active_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
if result ~= 0
  return;
end
func_ = mfilename;

ch_type = vb_meginfo_get_channel_type_extra(meginfo);

if active_swt
  ch_active = vb_meginfo_get_channel_active_extra(meginfo);
  ch_type = ch_type(ch_active == 1);
end

switch ex_type
  case  0 % new spec (except refmg_ch)
    % yokogawa spec
    target_ch = find(ch_type <=0);

  case  10 % old spec (include refmg_ch)
    target_ch = ch_type;

  otherwise
    fprintf('(%s)unexpected ex_type : %d\n', func_, ex_type);
    result = 2;
    return;
end

Nchannel = length(target_ch);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [meginfo, ex_type, active_swt, result] = ...
  inner_check_arguments(meginfo, ex_type, active_swt)

func_ = mfilename;
result = 0;

if isempty(meginfo)
  fprintf('(%s)meginfo is a required parameter\n', func_);
  result = 1;
  return;
end

if ~isfield(meginfo, 'ExtraChannelInfo')
  fprintf('(%s)meginfo must have ExtraChannelInfo field\n', func_);
  result = 1;
  return;
end

if isempty(ex_type)
  ex_type = 0;
end

if ~strcmp(meginfo.device, 'YOKOGAWA')
  fprintf('(%s)sorry, we have NOT supported except for YOKOGAWA\n', func_);
  ex_type = 10;
end

if isempty(active_swt)
  active_swt = false;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
