function [meg_data, ch_info] = vb_megfile_make_trial_data(meg_file, proc_spec)
% Make trial data files of MEG after extracting trials from continuous raw data
% [usage]
%   [meg_data, ch_info] = vb_megfile_make_trial_data(meg_file, proc_spec)
%
% [input]
%    meg_file : <required> MEG-MAT file as base (with path)
%   proc_spec : <required> <<struct>> struct defined process specifications
%             :  fields are as follows:
%      .root_dir  : <optional> data directory for new meg file
%                 : if not specified, directory of meg_file is used
%        .trigger : <required> list of sample of trigger [1 x Ntrial]
%  .pretrigger_ms : <required> pretrigger [msec]
% .posttrigger_ms : <required> posttrigger [msec]
%    .sample_freq : <requied> sampling frequency [Hz]
%       .new_file : <optional> new MEG file (without path)
%                 : this will be made at 'root_dir'
%                 : if this is not specified, data is not stored.
%    .new_bin_dir : <optional> external data directory where chopped data 
%                 :  will be stored.
%                 :  This is relative path from '.new_file'
%                 :  if this is not specified, data is stored internally
%                 :  (bexp etc.)
%
% [output]
%   meg_data : chopped data [Nchannel x Nsample x Ntrial]
%    ch_info : <<struct>> channel information of loaded data
%            :  .Active [Nchannel x 1]
%            :  .Name   [Nchannel x 1]
%            :  .ID     [Nchannel x 1]
%            :  .Type   [Nchannel x 1]
%
% [note]
%   @see vb_msrmnt_make_trial_data.m
%
% [history]
%   2009-07-24 (Sako) initial version
%   2009-09-01 (Sako) substituted vb_load_measurement_info for vb_load_meg_info
%   2011-06-02 (Sako) modified according to the new data format
%   2018-08-22 (Takeda) Modified so that cond_id is inherited 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('meg_file', 'var'), meg_file = ''; end
if ~exist('proc_spec', 'var'), proc_spec = []; end
[meg_file, proc_spec] = inner_check_arguments(meg_file, proc_spec);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
meg_data = [];
ch_info = [];

fs0  = proc_spec.sample_freq;

meg_info = vb_load_measurement_info(meg_file);
fs  = vb_info_get_sampling_frequency(meg_info);

% Recalculate trial time index according to 'Pre/Posttrigger'
% if Sample Frequency is different from raw data
% Convert trigger time index for fs [Hz]
frate = (fs/fs0);

% --- initial trig is loaded from trig_file
trigger = round( proc_spec.trigger * frate );

cond_id = proc_spec.cond_id;

% Sample number for Pre/Post period in data file
pretrigger  = ceil(proc_spec.pretrigger_ms  * (fs/1000));
posttrigger = ceil(proc_spec.posttrigger_ms * (fs/1000));

std_sample_list = (-pretrigger:posttrigger)';
n_sample = length(std_sample_list);
n_trial  = length(trigger);

trig_info.Trigger = trigger;
trig_info.Pretrigger = pretrigger;
trig_info.Posttrigger = posttrigger;
[tied_sample] = vb_loadspec_make_trial_sample(trig_info);
% --- tied_sample is the array of from and to [Ntrial x 2]

ix_trial = zeros(n_sample, n_trial);
for i_trial = 1:n_trial
  ix_trial(:,i_trial) = tied_sample(i_trial,1):tied_sample(i_trial,2);
end

% ---
% --- If new binary directory is given, store chopped data as the new
% --- binary data externally.
% --- If it is not given, store chopped data internally (bexp)
% ---
if isempty(proc_spec.new_file)
  new_bin_dir = '';
  save_externally = false;
else
  
  if ~isfield(proc_spec, 'new_bin_dir') || isempty(proc_spec.new_bin_dir)
    new_bin_dir = '';
    save_externally = false;

  else
    save_externally = true;

    new_bin_dir = fullfile(proc_spec.root_dir, proc_spec.new_bin_dir);
    if exist(new_bin_dir, 'dir') ~= 7
      vb_mkdir(new_bin_dir);
    end
    precision = vb_meginfo_get_precision(meg_info);
  end
end

if save_externally

  ch_labels = vb_megfile_get_channel_label_whole(meg_file);
  ch_len    = size(ch_labels, 1);

  fprintf('\n...now on chopping data ...\n');
  LINE_LEN = 30;
  
  for i_ch = 1:ch_len
    load_spec.ChannelName = {ch_labels{i_ch}};
    load_spec.ChannelSwitch = true;
    load_spec.ChannelType = 'ALL';
    load_spec.ActiveChannel = 0;
    load_spec.Trigger = trigger;
    load_spec.Pretrigger = pretrigger;
    load_spec.Posttrigger = posttrigger;
    chopped_data = vb_load_meg_data(meg_file, load_spec);

    % --- re-stored filtered data
    new_file = sprintf('%s/%s.ch.meg.dat', new_bin_dir, ch_labels{i_ch});
    fid = fopen(new_file, 'wb');
    if fid == -1
      warning('(%s) cannot open file : %s\n', mfilename, new_file);
      continue;
    end
    fwrite(fid, chopped_data, precision);
    fclose(fid);
    
    fprintf('.');
    if ~rem(i_ch, LINE_LEN), fprintf('\n'); end
  end
  
  % --- set empty for internal data buffers
  bexp     = [];
  bexp_ext = [];
  refmg    = [];

else
  % save MEG data as internal variable

  % --- (1)
  % --- load for every 'MEG' channels
  ch_labels = vb_megfile_get_channel_label_meg(meg_file);
  ch_id     = vb_meginfo_get_channel_index_meg(meg_info);
  ch_len    = size(ch_labels, 1);

  fprintf('\n...now on processing MEG data ...\n');
  
  bexp = zeros(ch_len, n_sample, n_trial);

  for i_ch = 1:ch_len
    load_spec.ChannelName = {ch_labels{i_ch}};
    load_spec.ChannelSwitch = true;
    load_spec.ChannelType = 'MEG';
    load_spec.ActiveChannel = 0;
    
    load_spec.Trigger = trigger;
    load_spec.Pretrigger = pretrigger;
    load_spec.Posttrigger = posttrigger;
    bexp(i_ch,:,:) = vb_load_meg_data(meg_file, load_spec);
    
    fprintf('.');
  end
  
  % --- (2)
  % --- load for every 'EXT' channels
  ch_labels = vb_megfile_get_channel_label_extra(meg_file);
  ch_id     = vb_meginfo_get_channel_index_extra(meg_info);
  ch_len    = size(ch_labels, 1);

  fprintf('\n...now on processing EXT data ...\n');

  bexp_ext = zeros(ch_len, n_sample, n_trial);

  for i_ch = 1:ch_len
    load_spec.ChannelName = {ch_labels{i_ch}};
    load_spec.ChannelSwitch = true;
    load_spec.ChannelType = 'EXT';
    load_spec.ActiveChannel = 0;
    load_spec.Trigger = trigger;
    load_spec.Pretrigger = pretrigger;
    load_spec.Posttrigger = posttrigger;
    bexp_ext(i_ch,:,:) = vb_load_meg_data(meg_file, load_spec);

    fprintf('.');
  end

  % --- (3)
  % --- load for every 'REF' channels
  ch_labels = vb_megfile_get_channel_label_refmg(meg_file);
  ch_id     = vb_meginfo_get_channel_index_refmg(meg_info);
  ch_len    = size(ch_labels, 1);

  fprintf('\n...now on processing REF data ...\n');

  refmg = zeros(ch_len, n_sample, n_trial);

  for i_ch = 1:ch_len
    load_spec.ChannelName = {ch_labels{i_ch}};
    load_spec.ChannelSwitch = true;
    load_spec.ChannelType = 'REF';
    load_spec.ActiveChannel = 0;
    load_spec.Trigger = trigger;
    load_spec.Pretrigger = pretrigger;
    load_spec.Posttrigger = posttrigger;
    refmg(i_ch,:,:) = vb_load_meg_data(meg_file, load_spec);

    fprintf('.');
  end
end
fprintf('\n');


% --- POSTERIORI PROCESS

% ----- make new meg.mat file
meg = load(meg_file);
new_meginfo = meg.MEGinfo;
meg.ParentInfo = meg.MEGinfo;

if ~isempty(proc_spec.new_file)
  % --- update fields
  if save_externally
    new_meginfo.saveman.data_dir = proc_spec.new_bin_dir;
  else
    new_meginfo.saveman = [];
  end

  for i_trial = 1:n_trial
    new_meginfo.Trial(i_trial).number = i_trial;
    new_meginfo.Trial(i_trial).sample = ix_trial(:,i_trial);

    new_meginfo.Trial(i_trial).Active = 1;
%     Trial.number = i_trial;
%     Trial.sample = ix_trial(:,i_trial);
% 
%     Trial.Active = 1;
%     new_meginfo.Trial(i_trial) = Trial;
  end
  new_meginfo.Trial = new_meginfo.Trial';

  new_meginfo.ActiveTrial = ones(n_trial,1);
  new_meginfo = vb_meginfo_set_acqtype(new_meginfo, 'Evoked_Raw');

  new_meginfo.Nsample = n_sample;
  new_meginfo.Nrepeat = n_trial;
  new_meginfo.Pretrigger = pretrigger;
  new_meginfo.cond_id = cond_id;
    
  meg.MEGinfo = new_meginfo;

  meg.bexp = bexp;
  meg.bexp_ext = bexp_ext;
  meg.refmg = refmg;

  fprintf('\n');
  
  save_file = fullfile(proc_spec.root_dir, proc_spec.new_file);
  vb_save_struct(save_file, meg);
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [meg_file, proc_spec] = inner_check_arguments(meg_file, proc_spec)
func_ = mfilename;

% --- check meg_file
if isempty(meg_file)
  error('(%s) meg_file is a required parameter', func_);
end

if exist(meg_file, 'file') ~= 2
  error('(%s) cannot find meg_file : %s', func_, meg_file);
end

% --- check proc_spec ---
%     .root_dir
%     .trigger
%     .pretrigger_ms
%     .posttrigger_ms
%     .sample_freq
%     .new_file
%     .new_bin_dir
% -----------------------
if isempty(proc_spec)
  error('(%s) proc_spec is a required parameter', func_);
end

% --- required fields

% ----- proc_spec.trigger
if ~isfield(proc_spec, 'trigger') || isempty(proc_spec.trigger)
  error('(%s) trigger is a required field of proc_spec', func_);
end

% ----- proc_spec.pretrigger_ms
if ~isfield(proc_spec, 'pretrigger_ms') || isempty(proc_spec.pretrigger_ms)
  error('(%s) pretrigger_ms is a required field of proc_spec', func_);
end

% ----- proc_spec.posttrigger_ms
if ~isfield(proc_spec, 'posttrigger_ms') || isempty(proc_spec.posttrigger_ms)
  error('(%s) posttrigger_ms is a required field of proc_spec', func_);
end

% ----- proc_spec.sample_freq
if ~isfield(proc_spec, 'sample_freq') || isempty(proc_spec.sample_freq)
  error('(%s) sample_freq is a required field of proc_spec', func_);
end


% --- optional fields

% ----- get path of meg_file as a default path
[fpath] = vb_get_file_parts(meg_file);

% ----- proc_spec.root_dir
if ~isfield(proc_spec, 'root_dir') || isempty(proc_spec.root_dir)
	proc_spec.root_dir = fpath;
end

% ----- proc_spec.new_file
if ~isfield(proc_spec, 'new_file')
  proc_spec.new_file = '';
end

% ----- proc_spec.new_bin_dir
if ~isfield(proc_spec, 'new_bin_dir')
  proc_spec.new_bin_dir = '';
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
