function [pick, Qpick, CoilWeight, Vcenter] = ...
  vb_megfile_load_specified_sensor(megfile, ch_name, active_swt)
% return sensor data which are specified by ch_name
% [usage]
%   [pick, Qpick, CoilWeight, Vcenter] = ...
%     vb_megfile_load_specified_sensor(megfile, ch_name, active_swt)
% [input]
%      megfile : <required> <<file>> MEG-MAT file
%      ch_name : <optional> [Nch x 1] list of channel name
%              :  if this is empty, target is all the channels
%   active_swt : <optional> <<boolean>> active switch [false]
%              :   true : target channels are only active
%              :  false : target channels are all the channels
% [output]
%         pick : n-th coil position  (Right-hand SPM coordinate) [m]
%              :  [Ncoil x 3]
%        Qpick : n-th coil direction (Right-hand SPM coordinate)
%              :  [Ncoil x 3]
%   CoilWeight : n-th coil weight for m-th channel [Nchannel x Ncoil]
%              : basis(channel,dipole) = CoilWeight * basis(coil,dipole)
%      Vcenter : Center of spherical head model [m] [1 x 3]
% [note]
%   @see vb_load_sensor.m
%   @see vb_meginfo_get_channel_index_meg.m
%   @see vb_meginfo_get_channel_index_extra.m
% [history]
%   2008-08-25 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('megfile', 'var'), megfile = ''; end
if ~exist('ch_name', 'var'), ch_name = []; end
if ~exist('active_swt', 'var'), active_swt = []; end
[megfile, ch_name, active_swt] = ...
  inner_check_arguments(megfile, ch_name, active_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
[org_pick, org_Qpick, org_CoilWeight, org_Vcenter] = ...
  vb_load_sensor(megfile, 'MEG', active_swt);

n_ch   = size(org_CoilWeight,1);

meginfo = vb_load_meg_info(megfile);
ch_idx = vb_meginfo_get_channel_index_meg(meginfo, ch_name, active_swt);
coil_idx = [ch_idx; ch_idx+n_ch];

pick = org_pick(coil_idx, :);
Qpick = org_Qpick(coil_idx, :);
CoilWeight = org_CoilWeight(ch_idx, coil_idx);
Vcenter = org_Vcenter;
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [megfile, ch_name, active_swt] = ...
  inner_check_arguments(megfile, ch_name, active_swt)
func_ = mfilename;
if isempty(megfile)
  error('(%s)megfile is a required parameter', func_);
end

if exist(megfile, 'file') ~= 2
  error('(%s)cannot find megfile : %s', func_, megfile);
end

if isempty(ch_name)
  % require no action
end

if isempty(active_swt)
  % default target is all the channels
  active_swt = false;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %
