function [data, channel_info] = vb_megfile_load_data(megfile, ...
  loadspec, new_file, return_swt, verbose_swt)
% load specified data from MEG-MAT file
%
% [usage]
%   [data, channel_info] = vb_megfile_load_data(megfile, ...
%             loadspec, new_file, return_swt, verbose_swt)
%
% [input]
%       megfile : <required> <<file>> MEG-MAT file
%
%      loadspec : <required> <<struct>> loading instruction
%      .ChannelName : <optional> channel name list [Nch x 1] []
%                   :  name(string) list of  target channel
%                   :  The way to be used this depends on the
%                   :  "ChannelSwitch" field
%                   :  e.g.
%                   :   {'1';'2';'3';'4'}
%                   :  If this is empty, all the channels are
%                   :  specified
%    .ChannelSwitch : <optiolal> <<boolean>> [true] | false
%                   :   Which is "ChannelName", to read, or not?
%                   :   [true] : to read
%                   :    false : to omit
%      .ChannelType : <optional> <<string>> channel type ['MEG']
%                   :  'MEG'       : MEG channel data
%                   :  'EXTRA'     : extra channel data
%                   :  'REFERENCE' : reference channel data
%                   :  'AXIAL'     : AxialGradioMeter (only MEG)
%                   :  'PLANAR'    : PlanarGradioMeter (only MEG)
%                   :  'ALL'       : 'MEG'+'EXTRA'+'REFERENCE'
%                   :  - case insensitive
%          .Trigger : <optional> [1]
%       .Pretrigger : <optional> [0]
%      .Posttrigger : <optional> [Nsample - 1]
%      .TrialNumber : <optional> trial number list [] [Ntrial x 1]
%                   :   "1" start
%                   :    e.g. [1;2;3]
%                   :    if this is empty, all the trials are specified
%    .ActiveChannel : <optional> <<boolean>> [false]
%                   :  active channel filter switch
%                   :    true) active channels
%                   :   false) all the channels
%      .ActiveTrial : <optional> <<boolean>> [false]
%                   :  active trial filter switch
%                   :  valid only when data type is Evoked_Raw
%                   :    true) active trials
%                   :   false) all the trials
%          .saveman : <optional> <<struct>>
%                   :  When you want to make a new file, loaded data will be
%                   :  stored in external directory.
%                   :  If this field is invalid or switch field is set false,
%                   :  data is stored internally.
%                   :    .data_dir  : data directory (relative path)
%                   :               : ['(name of new_file)_bin']
%                   :               : @see vb_device_make_data_dir.m
%                   :    .precision : data type ['float64']
%                   :    .switch    : <<optional>> switch to store externally
%                   :               :  [true]) store the data externally
%                   :               :   false) store the data internally
%
%      new_file : <optional> new MEG-MAT file name which is stored picked up
%               :  data and changed information []
%               :  if this is empty, new file will not be created
%    return_swt : <optional> <<boolean>> 
%               :  switch whether return loaded data or not
%               :  [true] | false
%   verbose_swt : <optional> <<boolean>> 
%               :  switch whether output verbose message or not
%               :  [true] | false
%
% [output]
%           data : loaded data [Nchannel x Nsample x Ntrial]
%   channel_info : <<struct>> channel information of loaded data
%                :  .Active [Nchannel x 1]
%                :  .Name   [Nchannel x 1]
%                :  .Type   [Nchannel x 1]
%                :  .ID     [Nchannel x 1]
%
% [note]
%   If internal data is not empty, extract data from it.
%   If they are empty, try to load data from binary files
%
%   See also
%     vb_megfile_load_data_external
%     vb_util_check_variable_in_matfile
%
% [history]
%   2007-07-11 (Sako) initial version
%   2008-02-15 (Sako) thoroughly revised
%   2008-04-17 (Sako) devided into two keeping data types internal and external
%   2008-04-30 (Sako) deleted unused inner function
%   2008-06-05 (Sako) changed interface (argument and return)
%   2009-09-02 (Sako) modified not to abort if read_ch_list is empty
%   2010-02-10 (Sako) improved checking internal data
%   2010-06-30 (Sako) supported storing data externally (only MEG)
%   2011-02-09 (Sako) set initial values to loadspec.saveman if necessary
%   2011-07-07 (Sako) allowed that the loaded data can be stored internally
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('megfile', 'var'), megfile = []; end
if ~exist('loadspec', 'var'), loadspec = []; end
if ~exist('new_file', 'var'), new_file = []; end
if ~exist('return_swt', 'var'), return_swt = []; end
if ~exist('verbose_swt', 'var'), verbose_swt = []; end
[megfile, loadspec, new_file, return_swt, VERBOSE] = ...
  inner_check_arguments(megfile, loadspec, new_file, return_swt, verbose_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
if isempty(loadspec)
  data = [];
  channel_info = [];
  return;
end

% --- check internal data
[state_bexp, const] = vb_util_check_variable_in_matfile(megfile, 'bexp');
[state_bexp_ext] = vb_util_check_variable_in_matfile(megfile, 'bexp_ext');
[state_refmg] = vb_util_check_variable_in_matfile(megfile, 'refmg');

if state_bexp ~= const.VALID ...
  && state_bexp_ext ~= const.VALID ...
  && state_refmg ~= const.VALID
  [data, channel_info] = vb_megfile_load_data_external(megfile, ...
    loadspec, new_file, return_swt, VERBOSE);
else
  [data, channel_info] = vb_megfile_load_data_internal(megfile, ...
    loadspec, new_file, return_swt, VERBOSE);
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [ ...
  megfile, loadspec, new_file, return_swt, VERBOSE] = ...
  inner_check_arguments(megfile, loadspec, new_file, return_swt, VERBOSE)

func_ = mfilename;

if isempty(megfile)
  error('(%s)megfile is a required parameter', func_);
end

if exist(megfile, 'file') ~= 2
  error('(%s)cannot find megfile : %s', func_, megfile);
end

meginfo = vb_megfile_load_meginfo(megfile);

% ----- check loadspec
loadspec = vb_loadspec_check(loadspec, megfile);


% ----- narrow channel by user-specified list and active filter
[read_ch_list] = vb_meginfo_get_read_channels(meginfo, ...
  loadspec.ChannelName, loadspec.ChannelSwitch, loadspec.ActiveChannel);

if isempty(read_ch_list)
%   error('(%s) read channels are empty', func_);
  fprintf('(%s) read channels are empty\n', func_);
  loadspec = [];
  return;
end
% ------- reset channel spec in loadspec
loadspec.ChannelName = read_ch_list;
loadspec.ChannelSwitch = true;

if ~isempty(new_file)
  if isfield(loadspec, 'saveman') && vb_saveman_get_switch(loadspec.saveman)
    loadspec.saveman.data_dir = vb_device_make_data_dir(new_file);
    loadspec.saveman.precision = 'float64';
  end
end

if isempty(return_swt)
  return_swt = true;
end

if isempty(new_file) && ~return_swt
  error('(%s)wrong combination - empty output file and no return', func_);
end

if isempty(VERBOSE)
  VERBOSE = true;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
