function [position, direction, coil_weight, center, channel_info] = ...
    vb_megfile_get_sensor_position_by_name(megfile, ch_name, active_swt)
% return information of sensor position specified by name
%
% [usage]
%   [position, direction, coil_weight, center] = ...
%     vb_megfile_get_sensor_position_by_name(megfile, ch_name, active_swt)
%
% [input]
%      megfile : <required> <<file>> MEG-MAT file
%      ch_name : <optional> <<cell array>> channel name list {''}
%              :  If this is invalid, return positions of 'MEG' channels
%   active_swt : <optional> <<boolean>> [false] only for MEG channels
%              :    true) return only active channels (MEG)
%              :   false) return all the channels (MEG)
%
% [output]
%     position : position data of sensors
%              :  (pick in MEG-MAT) [Ncoil x 3]
%    direction : direction data of sensors
%              :  (Qpick in MEG-MAT) [Ncoil x 3]
%  coil_weight : coil weight [Nchannel x Ncoil]
%       center : center coordinate [1 x 3]
% channel_info : <<struct>> channel information of loaded data
%                :  .Active [Nchannel x 1]
%                :  .Name   [Nchannel x 1]
%                :  .Type   [Nchannel x 1]
%                :  .ID     [Nchannel x 1]
%                empty return : including reference sensor name.
% [note]
%
%   See also
%     vb_megfile_get_sensor_position_by_type
%
% [history]
%   2011-07-26 (Sako) initial version
%   2017-03-06 (rhayashi) Add return : channel_info
%   2018-08-23 (Takeda) modified so that NEUROMAG is acceptable
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('megfile', 'var'), megfile = []; end
if ~exist('ch_name', 'var'), ch_name = ''; end
if ~exist('active_swt', 'var'), active_swt = []; end
[megfile, ch_name, active_swt] = ...
    inner_check_arguments(megfile, ch_name, active_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
channel_info = [];
meg_info = vb_megfile_load_meginfo(megfile);
ch_info_cur = vb_load_channel_info(megfile, 'USER', ch_name);

% --- for 'MEG' channels
ch_info_meg = vb_load_channel_info(megfile, 'MEG');
if isempty(ch_info_meg)
    active_swt = false;
end

if active_swt
    active_idx = find(ch_info_meg.Active == true);
    base_names = ch_info_meg.Name(active_idx, :);
else
    base_names = ch_info_meg.Name;
end

base_idx = vb_util_get_matching_label_index(base_names, ch_info_cur.Name);

if ~isempty(base_idx)
    target_idx = base_idx;
    
    load(megfile, 'pick', 'Qpick');
    c_weight = vb_meginfo_get_sensor_weight_meg(meg_info);
    
    target_coil_idx = find(sum(abs(c_weight(target_idx, :)), 1) > 0);
    
    pick_meg = pick(target_coil_idx, :);
    Qpick_meg = Qpick(target_coil_idx, :);
    c_weight_meg = c_weight(target_idx, target_coil_idx);
    
    channel_info.Active = ch_info_meg.Active(target_idx);
    channel_info.Name   = ch_info_meg.Name(target_idx);
    channel_info.Type   = ch_info_meg.Type(target_idx);
    channel_info.ID     = ch_info_meg.ID(target_idx);
else
    pick_meg = [];
    Qpick_meg = [];
    c_weight_meg = [];
end

% --- for 'REFERENCE' channels
pick_ref = [];
Qpick_ref = [];
c_weight_ref = [];

ch_info_ref = vb_load_channel_info(megfile, 'REFERENCE');
if ~isempty(ch_info_ref) && size(ch_info_ref.Name, 1)>0
    
    if active_swt
        active_idx = find(ch_info_ref.Active == true);
        base_names = ch_info_ref.Name(active_idx, :);
    else
        base_names = ch_info_ref.Name;
    end
    
    base_idx = vb_util_get_matching_label_index(base_names, ch_info_cur.Name);
    
    if ~isempty(base_idx)
        target_idx = base_idx;
        
        load(meg_file, 'ref_pick', 'ref_Qpick');
        pick_ref = ref_pick;
        Qpick_ref = ref_Qpick;
        c_weight_ref = vb_meginfo_get_sensor_weight_refmg(meg_info);
        
        target_coil_idx = find(sum(abs(c_weight_ref(target_idx, :)), 1) > 0);
        
        pick_ref = pick_ref(target_coil_idx, :);
        Qpick_ref = Qpick_ref(target_coil_idx, :);
        c_weight_ref = c_weight_ref(target_idx, target_coil_idx);
    end
end

% --- combine data
position = [pick_meg; pick_ref];
direction = [Qpick_meg; Qpick_ref];

len_meg_ch = size(c_weight_meg, 1);
len_meg_cl = size(c_weight_meg, 2);
len_ref_ch = size(c_weight_ref, 1);
len_ref_cl = size(c_weight_ref, 2);

len_ch = len_meg_ch + len_ref_ch;
len_coil = len_meg_cl + len_ref_cl;
coil_weight = zeros(len_ch, len_coil);

% --- coil weight for meg channels
coil_weight(1:len_meg_ch, 1:len_meg_cl) = c_weight_meg;

% --- coil_weight for reference channels
coil_weight((len_meg_ch+1):len_ch, (len_meg_cl+1):len_coil) = c_weight_ref;

% --- solve center coordinate
center = [];
if isfield(meg_info, 'Vcenter')
    center = meg_info.Vcenter;
end

return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [megfile, ch_name, active_swt] = ...
    inner_check_arguments(megfile, ch_name, active_swt)
func_ = mfilename;

if isempty(megfile)
    error('(%s) MEG-MAT file is a required parameter', func_);
end

if exist(megfile, 'file') ~= 2
    error('(%s)cannot find MEG-MAT file : %s', megfile);
end

[Measurement] = vb_load_device(megfile);

if ~strcmpi(Measurement, 'MEG')
    error('(%s)this file is not MEG file : %s', func_, megfile);
end

if ~iscell(ch_name)
    ch_name = {ch_name};
end

if isempty(ch_name{1})
    ch_info = vb_load_channel_info(megfile, 'MEG');
    ch_name = ch_info.Name;
end

if isempty(active_swt)
    active_swt = false;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
