function	[z,indxT,s] = vb_temporal_pca(x, Twin, smin)
% Temporal PCA
% [z,indxT,s] = vb_temporal_pca(x, Twin, smin)
% --- Input
% x : time series signal [NX x T]
% x(n,t) : n-th component signal at time t , t=1:T, n=1:NX
% Twin = [Nbck , Nfwd] : time shift length for backward and forward direction
%  if Twin is scalar, Nbck = Nfwd = Twin is assumed
%   smin : >= 1  : number of components for threshould PCA (muse be integer, default 40)
%           < 1  : contribution rate for threshould PCA
% --- Output
%  z : Time shifted signal after orthogonalization by PCA
%  indxT : time index for z
%  z(:,:) <-> x(:,indxT)
%  indxT = (1+Nbck):(T-Nfwd))
%  s : eigen value
%
% 2007-8-21 Masa-aki Sato
% 2015-4-20 Yusuke Fujiwara
% 2015-6-12 Yusuke Takeda

[NX,T] = size(x);

D  = sum(Twin) + 1;	% time shift length
TD = T - sum(Twin);	% Effective time length for process

if TD <= 0, error('Signal time length is too short'); end;

% subtract bias component
x = vb_repadd(x, - mean(x,2));

% --- Time shifted embedding index
%
%  z(:,t) = [ x(:,t-Nbck);
%                ...;
%             x(:,t);
%                ...;
%             x(:,t+Nfwd) ]

indxT = (1+Twin(1)):(T-Twin(2));
indxD = -Twin(1):Twin(2);
indx  = repmat( indxD(:),[1 TD]) + repmat(indxT,[D 1]);
indx  = indx(:);

X = reshape( x(:,indx), [NX*D TD]);

% --- SVD ---
% X = U * S * V , U:[N x N], S=diag(s):[N x N], V:[N x T]
%     U' * U = V * V' = eye(N)
% --- PCA ---
%  X * X'  = U * S^2 * U'
% (X * X') * U = U * S^2
%  V = S^(-2) * U' * X
% [U, D] = eig(A) : A * U = U * D , D = diag(d)

%[U,S,V] = svd( X * X' );

[U,S] = eig( X * X' );
s  = diag(S);
if smin<1
    ix = find( s/max(s) > smin );
else
    Npcs = length(s);
    ix = (Npcs - smin + 1):Npcs;
end

% --- Orthogonalize time shifted signal
z  = U(:,ix)' * X;

% Normalization
z  = vb_repmultiply( z, 1./sqrt(sum(z.^2,2)));

err1 = 1 - sum(s(ix))/sum(s);
err2 = max(max(abs( z*z' - eye(size(z,1)))));

fprintf('Dim of noise space = %d, ',length(ix))
fprintf('PCA err = %g, Ortho err = %g\n',err1,err2)

return

