#include "mex.h"
#include "mexutil.h"

/*
  Online calculation of highpass filter
  [y,z] = online_highpass(a,x,z)
% a  : Coefficient of highpass filter   [1 x 1]
% x  : Input signal                     [D x 1]
% z  : Lowpass internal state variable  [D x 1]
% --- Output
% y  : highpass signal                  [D x 1]

z = a * z + x * (1 - a);
y = x - z;

 Written by Masa-aki Sato 2009-6-20
*/

/* 
  Online calculation of highpass filter
z(j) = a * z(j) + x(j) * (1 - a);
y(j) = x(j) - z(j);
 */
void highpass_filter(double *x, double *y, double *z, double a, int d)
{
	double *zt, *yt, *xt;
	int j;
	
	xt = x;
	yt = y;
	zt = z;
	
  	for (j=0; j<d; j++) {
      	*zt = *zt * a + *xt * (1 - a);
      	*yt = *xt - *zt;
      	
      	xt++;
      	yt++;
      	zt++;
    }
}

/* The gateway routine */
void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[])
{
	double a,*x,*y,*z,*z0;
	int    D,t,T,n,j,x_cnt,y_cnt;
	mxArray *y_ptr,*z_ptr;
	
	/*  Check for proper number of arguments. */
	if(nrhs!=3) 
	  mexErrMsgTxt("3 inputs required.");
	if(nlhs!=2) 
	  mexErrMsgTxt("2 output required.");
	
	/*
	  Online calculation of highpass filter
	  [y,z] = online_highpass_filt(a,x,z)
	 a  : Coefficient of highpass filter   [1 x 1]
	 x  : Input signal                     [D x T]
	 z  : Lowpass internal state variable  [D x 1]
	 y  : highpass signal                  [D x T]
	
	for t=1:T
		z = a * z + x(:,t) * (1 - a);
		y(:,t) = x(:,t) - z;
	end
	
	*/
	
	/*  Get input value */
	a  = (double) mxGetScalar(prhs[0]);
	
	/*  Create a pointer to the inputs*/
	x  = mxGetPr(prhs[1]);
	z0 = mxGetPr(prhs[2]);
	
	/*  Get the dimensions of the matrix X */
	D = mxGetM(prhs[1]);
	T = mxGetN(prhs[1]);
	
	/*  Get the dimensions of the matrix Z */
	n = mxGetM(prhs[2]);
	t = mxGetN(prhs[2]);

	if(D!=n) 
	  mexErrMsgTxt("Dimension of X and Z do not match.");
	if(t!=1) 
	  mexErrMsgTxt("Dimension of Z should be D x 1.");
	
	/*  Set the output pointer to the output matrix. 
		plhs[0] = mxCreateDoubleMatrix(my,ny, mxREAL);
		Create uninitialized matrix for speed up
	*/
	plhs[0] = mxCreateDoubleMatrixE(D,T,mxREAL);
	plhs[1] = mxCreateDoubleMatrixE(D,1,mxREAL);
	
	/*  Create a C pointer to a copy of the output matrix. */
	y = mxGetPr(plhs[0]);
	z = mxGetPr(plhs[1]);
	
	/*  Initialize Z */
  	for (j=0; j<D; j++) {
      	z[j] = z0[j];
    }
    
	y_cnt = 0;
	x_cnt = 0;
	
	for (t=0; t<T; t++) {
    	
    	highpass_filter(x + x_cnt, y + y_cnt, z, a, D);
 		
		x_cnt += D;
		y_cnt += D;

	}
}
