function [SensorPosition, center, result, channel_info] = ...
  vb_eeginfo_get_sensor_position(EEGinfo, active_swt)
% get sensor position data from EEGinfo struct
% [usage]
%   [SensorPosition, center] = vb_eeginfo_get_sensor_position(EEGinfo, active_swt)
% [input]
%          EEGinfo : <required> EEGinfo struct data
%       active_swt : <optional> <<boolean>> [false] active filter switch
%                  :   true) return only active
%                  :  false) return all the channels
% [output]
%   SensorPosition : [N_sensor x 3] 3-D coordinates of sensor
%           center : [1 x 3] coordinate of center point
%           result : <<integer>> error code
%                  :  0) success
%                  :  1) bad eeginfo
%     channel_info : <<struct>> channel information of loaded data
%                    :  .Active [Nchannel x 1]
%                    :  .Name   [Nchannel x 1]
%                    :  .Type   [Nchannel x 1]
%                    :  .ID     [Nchannel x 1]% [note]
% [note]
%   @see vb_eeginfo_set_sensor_position
% [history]
%   2006-12-06 (Sako) initial version
%   2008-04-15 (Sako) added active_swt, result
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if ~exist('EEGinfo', 'var'), EEGinfo = []; end
if ~exist('active_swt', 'var'), active_swt = []; end
[EEGinfo, active_swt, result] = inner_check_arguments(EEGinfo, active_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
SensorPosition = [];
center = [];
channel_info = [];

if result ~= 0
  return;
end

if isfield(EEGinfo, 'Vcenter')
  center = EEGinfo.Vcenter;
end

if isfield(EEGinfo, 'Coord')
  SensorPosition = EEGinfo.Coord;
  % --- channel_info as a return value
  if isfield(EEGinfo, 'ChannelName')
      ch_info = vb_eeginfo_make_channel_info(EEGinfo, EEGinfo.ChannelName);
  else
      return;
  end
end

% --- active channel filter
if isempty(SensorPosition)
  % --- end here
  return;
end

if ~vb_info_active_channel_is_valid(EEGinfo)
  % do nothing any more
  return;
end

active_list = vb_info_get_active_channel(EEGinfo);
if length(active_list) ~= size(SensorPosition, 1)
  fprintf('(%s) different length - active list and sensor position\n', ...
    mfilename);
  return;
end

if active_swt
  % --- filtered sensor position
  active_ix = find(active_list == 1);
else
  % --- all the sensor position
  active_ix = [1:length(active_list)];
end

SensorPosition = SensorPosition(active_ix,:);
channel_info.Active = ones(length(active_ix), 1);
channel_info.Name   = ch_info.Name(active_ix);
channel_info.Type   = ch_info.Type(active_ix);
channel_info.ID     = ch_info.ID(active_ix);

return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [EEGinfo, active_swt, result] = ...
  inner_check_arguments(EEGinfo, active_swt)
func_ = mfilename;
result = 0;

if isempty(EEGinfo)
  fprintf('(%s)EEGinfo is a required parameter\n', func_);
  result = 1;
  return;
end

if isempty(active_swt)
  active_swt = false;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
