function [result] = vb_eegfile_replace_label(src_file, dst_file, rpl_type)
% replace channel labels of eegfile
% [usage]
%   [result] = vb_eegfile_replace_label(src_file, dst_file, rpl_type)
% [input]
%   src_file : <required> EEG-MAT file that will be converted
%   dst_file : <required> EEG-MAT file that will be newly made
%   rpl_type : <optional> replace type
%            :   0 ) EEG from Fp1 series to A1 series (Biosemi 64 ch default)
%            :  [1]) EEG from A1 series to Fp1 series (as the src is)
%            :   2 ) overwrite the Fp1 series label into existing labels
% [output]
%     result : ==0) success
%            : ~=0) error occured
% [note]
%   See also:
%   vb_posfile_replace_label
%   
% [history]
%   2012-11-14 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

func_ = mfilename;
result = 0; % success

% --- CHECK ARGUMENTS --- %
if ~exist('src_file', 'var'), src_file = ''; end
if ~exist('dst_file', 'var'), dst_file = ''; end
if ~exist('rpl_type', 'var'), rpl_type = []; end

[src_file, dst_file, rpl_type] = ...
  inner_check_arguments(src_file, dst_file, rpl_type);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
eeg = load(src_file);
if ~isfield(eeg, 'EEGinfo') || ~isfield(eeg.EEGinfo, 'ChannelName')
  fprintf('(%s) cannot find EEGinfo.ChannelName\n', func_);
  result = 1; % failure
  return;
end

switch rpl_type
  case 0
    new_labels = inner_fp1_to_a1(eeg.EEGinfo.ChannelName);

  case 1
    file_fld = [];
    if isfield(eeg.EEGinfo, 'File')
      if isfield(eeg.EEGinfo.File, 'DataDir') ...
          && ~isempty(eeg.EEGinfo.File.DataDir)
        file_fld = eeg.EEGinfo.File;
        
        f_path = fileparts(src_file);
        if isempty(f_path)
          f_path = './';
        end
        file_fld.eegfile_dir = f_path;
      end
    end
    new_labels = inner_a1_to_fp1(eeg.EEGinfo.ChannelName, file_fld);

  case 2
    new_labels = inner_set_fp1(eeg.EEGinfo.ChannelName);

  otherwise
    fprintf('(%s) unknown replace type : %d\n', func_, rpl_type);
    result = 1;
    return;
end

% overwrite labels
eeg.EEGinfo.ChannelName = new_labels;
vb_save_struct(dst_file, eeg);
return;
end
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [src_file, dst_file, rpl_type] = ...
  inner_check_arguments(src_file, dst_file, rpl_type)
func_ = mfilename;

if isempty(src_file)
  error('(%s) src_file is a required parameter', func_);
end

if exist(src_file, 'file') ~= 2
  error('(%s) cannot find src_file : %s\n', func_, src_file);
end

if isempty(dst_file)
  error('(%s) dst_file is a required parameter', func_);
end

if isempty(rpl_type)
  rpl_type = 1;
end
return;
end
%
% --- end of inner_check_arguments()


% --- inner_fp1_to_a1()
%
function new_labels = inner_fp1_to_a1(old_labels)

n_old = length(old_labels);
n_label = n_old;
new_labels = cell(n_label,1);

% --- copy old to new
for i_label = 1:n_label
  new_labels{i_label} = old_labels{i_label};
end

i_new = 1;

for i_old = 1:n_old
  if strcmp(old_labels{i_old}, 'Fp1'), new_labels{i_new} = 'A1';
  elseif strcmp(old_labels{i_old}, 'Fpz'), new_labels{i_new} = 'B1';
  elseif strcmp(old_labels{i_old}, 'Fp2'), new_labels{i_new} = 'B2';

  elseif strcmp(old_labels{i_old}, 'AF7'), new_labels{i_new} = 'A2';
  elseif strcmp(old_labels{i_old}, 'AF3'), new_labels{i_new} = 'A3';
  elseif strcmp(old_labels{i_old}, 'AFz'), new_labels{i_new} = 'B5';
  elseif strcmp(old_labels{i_old}, 'AF4'), new_labels{i_new} = 'B4';
  elseif strcmp(old_labels{i_old}, 'AF8'), new_labels{i_new} = 'B3';
  
  elseif strcmp(old_labels{i_old}, 'F7' ), new_labels{i_new} = 'A7';
  elseif strcmp(old_labels{i_old}, 'F5' ), new_labels{i_new} = 'A6';
  elseif strcmp(old_labels{i_old}, 'F3' ), new_labels{i_new} = 'A5';
  elseif strcmp(old_labels{i_old}, 'F1' ), new_labels{i_new} = 'A4';
  elseif strcmp(old_labels{i_old}, 'Fz' ), new_labels{i_new} = 'B6';
  elseif strcmp(old_labels{i_old}, 'F2' ), new_labels{i_new} = 'B7';
  elseif strcmp(old_labels{i_old}, 'F4' ), new_labels{i_new} = 'B8';
  elseif strcmp(old_labels{i_old}, 'F6' ), new_labels{i_new} = 'B9';
  elseif strcmp(old_labels{i_old}, 'F8' ), new_labels{i_new} = 'B10';
  
  elseif strcmp(old_labels{i_old}, 'FT7'), new_labels{i_new} = 'A8';
  elseif strcmp(old_labels{i_old}, 'FC5'), new_labels{i_new} = 'A9';
  elseif strcmp(old_labels{i_old}, 'FC3'), new_labels{i_new} = 'A10';
  elseif strcmp(old_labels{i_old}, 'FC1'), new_labels{i_new} = 'A11';
  elseif strcmp(old_labels{i_old}, 'FCz'), new_labels{i_new} = 'B15';
  elseif strcmp(old_labels{i_old}, 'FC2'), new_labels{i_new} = 'B14';
  elseif strcmp(old_labels{i_old}, 'FC4'), new_labels{i_new} = 'B13';
  elseif strcmp(old_labels{i_old}, 'FC6'), new_labels{i_new} = 'B12';
  elseif strcmp(old_labels{i_old}, 'FT8'), new_labels{i_new} = 'B11';
  
  elseif strcmp(old_labels{i_old}, 'T7' ), new_labels{i_new} = 'A15';
  elseif strcmp(old_labels{i_old}, 'C5' ), new_labels{i_new} = 'A14';
  elseif strcmp(old_labels{i_old}, 'C3' ), new_labels{i_new} = 'A13';
  elseif strcmp(old_labels{i_old}, 'C1' ), new_labels{i_new} = 'A12';
  elseif strcmp(old_labels{i_old}, 'Cz' ), new_labels{i_new} = 'B16';
  elseif strcmp(old_labels{i_old}, 'C2' ), new_labels{i_new} = 'B17';
  elseif strcmp(old_labels{i_old}, 'C4' ), new_labels{i_new} = 'B18';
  elseif strcmp(old_labels{i_old}, 'C6' ), new_labels{i_new} = 'B19';
  elseif strcmp(old_labels{i_old}, 'T8' ), new_labels{i_new} = 'B20';
  
  elseif strcmp(old_labels{i_old}, 'TP7'), new_labels{i_new} = 'A16';
  elseif strcmp(old_labels{i_old}, 'CP5'), new_labels{i_new} = 'A17';
  elseif strcmp(old_labels{i_old}, 'CP3'), new_labels{i_new} = 'A18';
  elseif strcmp(old_labels{i_old}, 'CP1'), new_labels{i_new} = 'A19';
  elseif strcmp(old_labels{i_old}, 'CPz'), new_labels{i_new} = 'A32';
  elseif strcmp(old_labels{i_old}, 'CP2'), new_labels{i_new} = 'B24';
  elseif strcmp(old_labels{i_old}, 'CP4'), new_labels{i_new} = 'B23';
  elseif strcmp(old_labels{i_old}, 'CP6'), new_labels{i_new} = 'B22';
  elseif strcmp(old_labels{i_old}, 'TP8'), new_labels{i_new} = 'B21';
  
  elseif strcmp(old_labels{i_old}, 'P9' ), new_labels{i_new} = 'A24';
  elseif strcmp(old_labels{i_old}, 'P7' ), new_labels{i_new} = 'A23';
  elseif strcmp(old_labels{i_old}, 'P5' ), new_labels{i_new} = 'A22';
  elseif strcmp(old_labels{i_old}, 'P3' ), new_labels{i_new} = 'A21';
  elseif strcmp(old_labels{i_old}, 'P1' ), new_labels{i_new} = 'A20';
  elseif strcmp(old_labels{i_old}, 'Pz' ), new_labels{i_new} = 'A31';
  elseif strcmp(old_labels{i_old}, 'P2' ), new_labels{i_new} = 'B25';
  elseif strcmp(old_labels{i_old}, 'P4' ), new_labels{i_new} = 'B26';
  elseif strcmp(old_labels{i_old}, 'P6' ), new_labels{i_new} = 'B27';
  elseif strcmp(old_labels{i_old}, 'P8' ), new_labels{i_new} = 'B28';
  elseif strcmp(old_labels{i_old}, 'P10'), new_labels{i_new} = 'B29';
  
  elseif strcmp(old_labels{i_old}, 'PO7'), new_labels{i_new} = 'A25';
  elseif strcmp(old_labels{i_old}, 'PO3'), new_labels{i_new} = 'A26';
  elseif strcmp(old_labels{i_old}, 'POz'), new_labels{i_new} = 'A30';
  elseif strcmp(old_labels{i_old}, 'PO4'), new_labels{i_new} = 'B31';
  elseif strcmp(old_labels{i_old}, 'PO8'), new_labels{i_new} = 'B30';

  elseif strcmp(old_labels{i_old}, 'O1' ), new_labels{i_new} = 'A27';
  elseif strcmp(old_labels{i_old}, 'Oz' ), new_labels{i_new} = 'A29';
  elseif strcmp(old_labels{i_old}, 'O2' ), new_labels{i_new} = 'B32';

  elseif strcmp(old_labels{i_old}, 'Iz' ), new_labels{i_new} = 'A28';
  end
  
  i_new = i_new + 1;
end
return;
end
%
% --- end of inner_fp1_to_a1()


% --- inner_a1_to_fp1()
%
function new_labels = inner_a1_to_fp1(old_labels, file_fld)

n_old = length(old_labels);
n_label = n_old;
new_labels = cell(n_label,1);

% --- copy old to new
for i_label = 1:n_label
  new_labels{i_label} = old_labels{i_label};
end
i_new = 1;

for i_old = 1:n_old
  if strcmp(old_labels{i_old}, 'A1')
    new_labels{i_new} = 'Fp1';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B1')
    new_labels{i_new} = 'Fpz';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B2')
    new_labels{i_new} = 'Fp2';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A2')
    new_labels{i_new} = 'AF7';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});
    
  elseif strcmp(old_labels{i_old}, 'A3')
    new_labels{i_new} = 'AF3';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});
    
  elseif strcmp(old_labels{i_old}, 'B5')
    new_labels{i_new} = 'AFz';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});
    
  elseif strcmp(old_labels{i_old}, 'B4')
    new_labels{i_new} = 'AF4';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B3')
    new_labels{i_new} = 'AF8';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A7' )
    new_labels{i_new} = 'F7';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A6' )
    new_labels{i_new} = 'F5';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A5' )
    new_labels{i_new} = 'F3';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A4' )
    new_labels{i_new} = 'F1';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B6' )
    new_labels{i_new} = 'Fz';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B7' )
    new_labels{i_new} = 'F2';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B8' )
    new_labels{i_new} = 'F4';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B9' )
    new_labels{i_new} = 'F6';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B10' )
    new_labels{i_new} = 'F8';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  
  elseif strcmp(old_labels{i_old}, 'A8')
    new_labels{i_new} = 'FT7';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A9')
    new_labels{i_new} = 'FC5';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A10')
    new_labels{i_new} = 'FC3';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A11')
    new_labels{i_new} = 'FC1';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B15')
    new_labels{i_new} = 'FCz';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B14')
    new_labels{i_new} = 'FC2';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B13')
    new_labels{i_new} = 'FC4';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B12')
    new_labels{i_new} = 'FC6';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B11')
    new_labels{i_new} = 'FT8';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A15' )
    new_labels{i_new} = 'T7';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A14' )
    new_labels{i_new} = 'C5';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A13' )
    new_labels{i_new} = 'C3';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A12' )
    new_labels{i_new} = 'C1';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B16' )
    new_labels{i_new} = 'Cz';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B17' )
    new_labels{i_new} = 'C2';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B18' )
    new_labels{i_new} = 'C4';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B19' )
    new_labels{i_new} = 'C6';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B20' )
    new_labels{i_new} = 'T8';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  
  elseif strcmp(old_labels{i_old}, 'A16')
    new_labels{i_new} = 'TP7';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A17')
    new_labels{i_new} = 'CP5';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A18')
    new_labels{i_new} = 'CP3';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A19')
    new_labels{i_new} = 'CP1';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A32')
    new_labels{i_new} = 'CPz';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B24')
    new_labels{i_new} = 'CP2';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B23')
    new_labels{i_new} = 'CP4';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B22')
    new_labels{i_new} = 'CP6';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B21')
    new_labels{i_new} = 'TP8';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  
  elseif strcmp(old_labels{i_old}, 'A24' )
    new_labels{i_new} = 'P9';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A23' )
    new_labels{i_new} = 'P7';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A22' )
    new_labels{i_new} = 'P5';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A21' )
    new_labels{i_new} = 'P3';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A20' )
    new_labels{i_new} = 'P1';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A31' )
    new_labels{i_new} = 'Pz';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B25' )
    new_labels{i_new} = 'P2';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B26' )
    new_labels{i_new} = 'P4';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B27' )
    new_labels{i_new} = 'P6';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B28' )
    new_labels{i_new} = 'P8';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B29')
    new_labels{i_new} = 'P10';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  
  elseif strcmp(old_labels{i_old}, 'A25')
    new_labels{i_new} = 'PO7';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A26')
    new_labels{i_new} = 'PO3';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A30')
    new_labels{i_new} = 'POz';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B31')
    new_labels{i_new} = 'PO4';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B30')
    new_labels{i_new} = 'PO8';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});


  elseif strcmp(old_labels{i_old}, 'A27' )
    new_labels{i_new} = 'O1';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'A29' )
    new_labels{i_new} = 'Oz';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  elseif strcmp(old_labels{i_old}, 'B32' )
    new_labels{i_new} = 'O2';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});


  elseif strcmp(old_labels{i_old}, 'A28' )
    new_labels{i_new} = 'Iz';
    inner_replace_chdata_label(file_fld, old_labels{i_old}, new_labels{i_new});

  end
  
  i_new = i_new + 1;
end
return;
end
%
% --- end of inner_a1_to_fp1()

% --- inner_set_fp1()
%
function base_labels = inner_sub_64_set_fp1(base_labels)

i_new = 1;

base_labels{i_new} = 'Fp1'; i_new = i_new + 1;
base_labels{i_new} = 'Fpz';i_new = i_new + 1;
base_labels{i_new} = 'Fp2';i_new = i_new + 1;
base_labels{i_new} = 'AF7';i_new = i_new + 1;
base_labels{i_new} = 'AF3';i_new = i_new + 1;
base_labels{i_new} = 'AFz';i_new = i_new + 1;
base_labels{i_new} = 'AF4';i_new = i_new + 1;
base_labels{i_new} = 'AF8';i_new = i_new + 1;
base_labels{i_new} = 'F7';i_new = i_new + 1;
base_labels{i_new} = 'F5';i_new = i_new + 1;
base_labels{i_new} = 'F3';i_new = i_new + 1;
base_labels{i_new} = 'F1';i_new = i_new + 1;
base_labels{i_new} = 'Fz';i_new = i_new + 1;
base_labels{i_new} = 'F2';i_new = i_new + 1;
base_labels{i_new} = 'F4';i_new = i_new + 1;
base_labels{i_new} = 'F6';i_new = i_new + 1;
base_labels{i_new} = 'F8';i_new = i_new + 1;
base_labels{i_new} = 'FT7';i_new = i_new + 1;
base_labels{i_new} = 'FC5';i_new = i_new + 1;
base_labels{i_new} = 'FC3';i_new = i_new + 1;
base_labels{i_new} = 'FC1';i_new = i_new + 1;
base_labels{i_new} = 'FCz';i_new = i_new + 1;
base_labels{i_new} = 'FC2';i_new = i_new + 1;
base_labels{i_new} = 'FC4';i_new = i_new + 1;
base_labels{i_new} = 'FC6';i_new = i_new + 1;
base_labels{i_new} = 'FT8';i_new = i_new + 1;
base_labels{i_new} = 'T7'; i_new = i_new + 1;
base_labels{i_new} = 'C5'; i_new = i_new + 1;
base_labels{i_new} = 'C3'; i_new = i_new + 1;
base_labels{i_new} = 'C1'; i_new = i_new + 1;
base_labels{i_new} = 'Cz'; i_new = i_new + 1;
base_labels{i_new} = 'C2'; i_new = i_new + 1;
base_labels{i_new} = 'C4'; i_new = i_new + 1;
base_labels{i_new} = 'C6'; i_new = i_new + 1;
base_labels{i_new} = 'T8'; i_new = i_new + 1;
base_labels{i_new} = 'TP7'; i_new = i_new + 1;
base_labels{i_new} = 'CP5'; i_new = i_new + 1;
base_labels{i_new} = 'CP3'; i_new = i_new + 1;
base_labels{i_new} = 'CP1'; i_new = i_new + 1;
base_labels{i_new} = 'CPz'; i_new = i_new + 1;
base_labels{i_new} = 'CP2'; i_new = i_new + 1;
base_labels{i_new} = 'CP4'; i_new = i_new + 1;
base_labels{i_new} = 'CP6'; i_new = i_new + 1;
base_labels{i_new} = 'TP8'; i_new = i_new + 1;
base_labels{i_new} = 'P9'; i_new = i_new + 1;
base_labels{i_new} = 'P7'; i_new = i_new + 1;
base_labels{i_new} = 'P5'; i_new = i_new + 1;
base_labels{i_new} = 'P3'; i_new = i_new + 1;
base_labels{i_new} = 'P1'; i_new = i_new + 1;
base_labels{i_new} = 'Pz'; i_new = i_new + 1;
base_labels{i_new} = 'P2'; i_new = i_new + 1;
base_labels{i_new} = 'P4'; i_new = i_new + 1;
base_labels{i_new} = 'P6'; i_new = i_new + 1;
base_labels{i_new} = 'P8'; i_new = i_new + 1;
base_labels{i_new} = 'P10'; i_new = i_new + 1;
base_labels{i_new} = 'PO7'; i_new = i_new + 1;
base_labels{i_new} = 'PO3'; i_new = i_new + 1;
base_labels{i_new} = 'POz'; i_new = i_new + 1;
base_labels{i_new} = 'PO4'; i_new = i_new + 1;
base_labels{i_new} = 'PO8'; i_new = i_new + 1;
base_labels{i_new} = 'O1'; i_new = i_new + 1;
base_labels{i_new} = 'Oz'; i_new = i_new + 1;
base_labels{i_new} = 'O2'; i_new = i_new + 1;
base_labels{i_new} = 'Iz'; i_new = i_new + 1;

return;
end
%
% --- end of inner_set_fp1()

% --- inner_replace_chdata_label()
%
function [result] = inner_replace_chdata_label(file_fld, old_label, new_label)
data_dir = [file_fld.eegfile_dir '/' file_fld.DataDir];

file_info = dir(data_dir);
n_file = length(file_info);

old_label_len = length(old_label);

for i_file = 1:n_file
  old_file = file_info(i_file).name;
  if strncmp(old_file, old_label, old_label_len)
    new_file = [new_label old_file((old_label_len+1):end)];
    movefile([data_dir '/' old_file], [data_dir '/' new_file]);
    break;
  end
end
return;
end
%
% --- end of inner_replace_chdata_label()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
