function [trigger_channel_idx, trigger_sample_idx] = ...
  vb_util_read_trigger_file(trigger_file, sample_freq)
% read trigger file and get trigger samples
% [usage]
%   [trigger_cannel_idx, trigger_sample_idx] = ...
%     vb_util_read_trigger_file(trigger_file, sample_freq)
% [input]
%   trigger_file : <required> trigger file
%    sample_freq : <conditionally required> sampling frequency
%                : this parameter is necessary when my_type is 'TIME'
% [output]
%   trigger_channel_idx : index list of trigger channel (empty data)
%    trigger_sample_idx : index of trigger samples
% [note]
%   trigger_file must have the struct 'trigger_spec' as follows
%   trigger_spec : <<struct>> dataset of trigger
%                :  - my_list : list of trigger (time or sample number)
%                :  - my_type : 'TIME' or 'SAMPLE'
%                :  - my_unit : 'MS' means milli seccond
%                :  - idx_type: 'ALL' or 'BEGIN' or 'END' ['BEGIN']
% [history]
%   2007-01-11 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENT --- %
if ~exist('trigger_file', 'var') trigger_file = []; end;
if ~exist('sample_freq', 'var') sample_freq = []; end;

[trigger_file, sampling_frequency, trigger_spec] = ...
  inner_check_argument(trigger_file, sample_freq);


% --- MAIN PROCEDURE --------------------------------------------------------- %
%
trigger_channel_idx = [];

switch trigger_spec.my_type
  case  'SAMPLE'
    trigger_sample_idx = ...
      inner_set_sample_index(trigger_spec.idx_type, trigger_spec.my_list);
        
  case  'TIME'
    switch trigger_spec.my_unit
      case  'MS'
        if ~isempty(sampling_frequency)
          [msec_time, sample_idx] = vb_util_sample2msectime2sample( ...
            [], trigger_spec.my_list, sampling_frequency);
          trigger_sample_idx = ...
            inner_set_sample_index(trigger_spec.idx_type, sample_idx);
        end
      otherwise
        error('unknown trigger_spec.unit : %s', trigger_spec.my_unit);
    end
        
  otherwise
    % as a sample list for now
    trigger_sample_idx = ...
      inner_set_sample_index(trigger_spec.idx_type, trigger_spec.my_list);

end % end of switch trigger_spec.my_type

%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_argument()
%
function [trigger_file, sample_freq, trigger_spec] = ...
  inner_check_argument(trigger_file, sample_freq)

if isempty(trigger_file)
  error('trigger_file is a required parameter');
end

if exist(trigger_file, 'file') ~= 2
  error('cannot read trigger_file : %s', trigger_file)
end

% initialize
trigger_spec.my_list = [];
trigger_spec.my_type = [];
trigger_spec.my_unit = [];
trigger_spec.idx_type = [];

load(trigger_file);

if ~exist('my_list', 'var')
  error('trigger_file ''%s'' does not have trigger list data', trigger_file);
end

if ~exist('my_type', 'var'), my_type = 'SAMPLE'; end;
if ~exist('my_unit', 'var'), my_unit = 'MS'; end;
if ~exist('idx_type', 'var'), idx_type = 'BEGIN'; end;

if strcmp(my_type, 'TIME') && isempty(sample_freq)
  error('''sample_freq'' is necessary when type is ''TIME''');
end

trigger_spec.my_list = my_list; end;
trigger_spec.my_type = my_type; end;
trigger_spec.my_unit = my_unit; end;
trigger_spec.idx_type = idx_type; end;
%
% --- end of inner_check_argument()

% --- inner_set_sample_index()
%
function sample_idx = inner_set_sample_index(index_type, this_index)

% this_index size must be [1 x N] in this function
this_index = vb_util_arrange_list(this_index, 1);

switch index_type
  case  'ALL'
    sample_idx.all_idx = this_index;
  case  'BEGIN'
    sample_idx.beg_idx = this_index;
  case  'END'
    sample_idx.end_idx = this_index;
  otherwise
    error('unknown index type: %s', index_type);
end
%
% --- end of inner_set_sample_index()
%%% END OF FILE %%%
