function [trial_times, pretrigger, posttrigger] = vb_read_trial_spec( ...
  info, trial_spec)
% derive a set of trial time ([from to]) from trial_spec
% [usage]
%   [trial_times, pretrigger, posttrigger] = vb_read_trial_spec( ...
%     info, trial_spec)
% [input]
%          info : <required> <<struct>> (EEGinfo or MEGinfo)
%               : information of target measurement_data
%    trial_spec : <required> <<struct>>
%               : specification of this trial. The details are below.
% [output]
%   trial_times : array of combinations of trigger_time, time_from and time_to
%               : [Ntrial x 3]
%    pretrigger : number of sample before trigger
%   posttrigger : number of sample after trigger
% [note]
%   <struct> trial_spec
%     trigger_list : sample number of triggers
%      pre_trigger : time before trigger point [ms] - (A)
%     post_trigger : time after trigger point [ms]  - (B)
%     segmented_ti : segmented time interval [ms]   - (C)
%     segment_rule : priority of segment rule (1 or 2 or 3)
%                  : Necessary parameters are two to segment.
%                  : If three parameters above are given,
%                  : this rule would be applied.
%                  :   +-----+-----+-----+-----+
%                  :   |     | (A) | (B) | (C) |
%                  :   +-----+-----+-----+-----+
%                  :   | (A) | --- | (1) | (2) |
%                  :   +-----+-----+-----+-----+
%                  :   | (B) | --- | --- | (3) |
%                  :   +-----+-----+-----+-----+
%                  :   | (C) | --- | --- | --- |
%                  :   +-----+-----+-----+-----+
% [history]
%   2006-12-21 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('info', 'var')             info = []; end;
if ~exist('trial_spec', 'var') trial_spec = []; end;

[info,trial_spec] = inner_check_arguments(info,trial_spec);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
% if trial_spec is essentially empty, return empty array
if inner_trial_spec_is_empty(trial_spec)
  trial_times = [];
  pretrigger = 0;
  posttrigger = 0;
  return;
end

% solve segment rule
if ~isfield(trial_spec, 'segment_rule')
  trial_spec.segment_rule = 1;
end

state = inner_trial_spec_get_parameter_state(trial_spec);

sampling_frequency = vb_info_get_sampling_frequency(info);

[trial_times, pretrigger, posttrigger] = inner_calc_timeset_by_state( ...
  state, trial_spec, sampling_frequency);
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_trial_spec_is_empty()
% inner function to judge whether trial_spec would be empty or not
% [usage]
%   result = inner_trial_spec_is_empty(trial_spec)
% [input]
%   trial_spec : trial_spec struct
% [output]
%       result : true or false
% [note]
%   If a structure of trial_spec change, this function must be modified.
% [history]
%   2006-12-15 (Sako) initial version
function result = inner_trial_spec_is_empty(trial_spec)

result = true;
if ~exist('trial_spec', 'var') ...
    || isempty(trial_spec)
  return;
  
elseif ~isfield(trial_spec, 'trigger_list') ...
    || isempty(trial_spec.trigger_list)
  return;
  
elseif (~isfield(trial_spec, 'pre_trigger') ...
      || isempty(trial_spec.pre_trigger)) ...
    && (~isfield(trial_spec, 'post_trigger') ...
      || isempty(trial_spec.post_trigger))
  return;
  
elseif (~isfield(trial_spec, 'post_trigger') ...
      || isempty(trial_spec.post_trigger)) ...
    && (~isfield(trial_spec, 'segmented_ti') ...
      || isempty(trial_spec.segmented_ti))
  return;
  
elseif (~isfield(trial_spec, 'segmented_ti') ...
      || isempty(trial_spec.segmented_ti)) ...
    && (~isfield(trial_spec, 'pre_trigger') ...
      || isempty(trial_spec.pre_trigger))
  return;
  
else
  result = false;
  return;
end
%
% --- end of inner_trial_spec_is_empty()

% --- inner_trial_spec_get_parameter_state()
% inner function to check parameter state
% [usage]
%   state = inner_trial_spec_get_parameter_state(trial_spec)
% [input]
%   trial_spec : trial_spec struct
% [output]
%        state : sum of state as follows
%              :  1) pre_trigger
%              :  2) post_trigger
%              :  4) segmented_ti
% [note]
%   Prior condition is that trial_spec is not 'EMPTY'.
% [history]
%   2006-12-15 (Sako) initial version
function state = inner_trial_spec_get_parameter_state(trial_spec)
state = 0;
if isfield(trial_spec, 'pre_trigger') && ~isempty(trial_spec.pre_trigger)
  state = state + 1;
end;

if isfield(trial_spec, 'post_trigger') && ~isempty(trial_spec.post_trigger)
  state = state + 2;
end;

if isfield(trial_spec, 'segmented_ti') && ~isempty(trial_spec.segmented_ti)
  state = state + 4;
end;
%
% --- end of inner_trial_spec_get_parameter_state()


% --- inner_calc_timeset_by_state()
% inner function to calculate trial time set by state
% [usage]
% [input]
% [output]
% [note]
% [history]
%   2006-12-15 (Sako) initial version
function [trial_times, pre, post] = inner_calc_timeset_by_state( ...
  state, trial_spec, s_freq)

trigger_list = trial_spec.trigger_list;

switch  state
  case  7   % all paramegers
    switch trial_spec.segment_rule
      case  1 % pre_trigger & post_trigger
        real_state = 3;        
      case  2 % pre_trigger & segmented_ti
        real_state = 5;
      case  3 % post_trigger & segmented_ti
        real_state = 6;
      otherwise
        warning('undefined segment_rule : %d\n', trial_spec.segment_rule);
        state = 3;
    end
    [trial_times, pre, post] = inner_calc_timeset_by_state( ...
      real_state, trial_spec, s_freq );

  case  6   % post_trigger & segmented_ti
    post = trial_spec.post_trigger;
    pre = trial_spec.segmented_ti - post;
    trial_times = inner_clip_timeset(pre, post, trigger_list, s_freq);
    
  case  5   % segmented_ti & pre_trigger
    pre = trial_spec.pre_trigger;
    post = trial_spec.segmented_ti - pre;
    trial_times = inner_clip_timeset(pre, post, trigger_list, s_freq);
    
  case  3   % pre_trigger & post_trigger
    pre = trial_spec.pre_trigger;
    post = trial_spec.post_trigger;
    trial_times = inner_clip_timeset(pre, post, trigger_list, s_freq);
    
  otherwise
    error('unknown state number : %d', state);
end
%
% --- end of inner_calc_timeset_by_state()


% --- inner_clip_timeset()
% inner function to clip trial time set
% [usage]
%   trial_times = inner_clip_timeset(pre, post, trigger_list, EEGinfo)
% [input]
%   pre : pre-trigger
%   post : post-trigger
%   trigger_list : list of trigger samples
%   s_freq : sampling frequency
% [output]
%   trial_times : array of trial time set (trigger, from&to) [Ntrial x 3]
% [note]
% [history]
%   2006-12-15 (Sako) initial version
function trial_times = inner_clip_timeset(pre, post, trigger_list, s_freq)
trial_times = [];
fprintf('inner_clip_timeset(%d,%d, ,%d)\n', pre, post, s_freq);

[pre_time, sample] = vb_util_sample2msectime2sample(pre, [], s_freq);
[post_time, sample] = vb_util_sample2msectime2sample(post, [], s_freq);

trigger_list = sort(trigger_list);
trigger_num = size(trigger_list,1);
for n_trial = 1:trigger_num
  [trigger_time, sample_list] = vb_util_sample2msectime2sample( ...
    trigger_list(n_trial), [], s_freq);
  from_t = trigger_time - pre_time;
  to_t = trigger_time + post_time;
  trial_times = [trial_times; [trigger_time, from_t, to_t]];
end
%
% --- end of inner_clip_timeset()

% --- inner_check_arguments()
% check argument
%
function [info, trial_spec] = inner_check_arguments(info, trial_spec)
if isempty(info)
  error('info is a required parameter');
end

if isempty('trial_spec')
  error('trial_spec is a required parameter');
end
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
