function [result] = vb_posfile_remake_eegfile(pos_file, eeg_file, new_file)
% remake EEG-MAT file by loading posfile
% [usage]
%   [result] = vb_posfile_remake_eegfile(pos_file, eeg_file, new_file)
% [input]
%   pos_file : <required> <<file>> POS-MAT file
%   eeg_file : <required> <<file>> EEG-MAT file
%   new_file : <optional> new file name if you want [(update eeg_file)]
% [output]
%     result : result code
%            :  0) no problem
%            : >1) something wrong
%            :      1) bad pos_file
%            :      2) bad eeg_file
% [note]
%   
% [history]
%   2009-08-05 (Sako) initial version
%   2011-06-01 (Sako) modified according to the new data format
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('pos_file', 'var'), pos_file = ''; end
if ~exist('eeg_file', 'var'), eeg_file = ''; end
if ~exist('new_file', 'var'), new_file = ''; end
[pos_file, eeg_file, new_file, result] = ...
  inner_check_arguments(pos_file, eeg_file, new_file);

if result ~= 0
  return;
end

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
cur_eeg = load(eeg_file);

pos_info = vb_posfile_get_posinfo(pos_file);
% --- POS_INFO ------------------------------------
%     SensorPosition: [96x3 double]
%          CoordType: 'SPM_Right_m'
%       ChannelLabel: {96x1 cell}
%            Vcenter: [1 x 3]
%            Vradius: [x1]
% -------------------------------------------------

eeg_info = vb_load_measurement_info(eeg_file);


[eeg_idx, pos_idx] = ...
  vb_util_get_index(eeg_info.ChannelName, pos_info.ChannelLabel);

% --- EEG channels
n_channel = length(eeg_idx);

old_ch_name = {};
new_ch_name = {};
if isfield(eeg_info, 'ChannelName')
  old_ch_name = eeg_info.ChannelName;
  new_ch_name = old_ch_name(eeg_idx);

  eeg_info.ChannelName = new_ch_name;
end

old_ch_id = [];
new_ch_id = [];
if isfield(eeg_info, 'ChannelID')
  old_ch_id = eeg_info.ChannelID;
  new_ch_id = old_ch_id(eeg_idx);
  eeg_info.ChannelID   = new_ch_id;
end

if isfield(eeg_info, 'ChannelInfo')
  ch_info.Active = eeg_info.ChannelInfo.Active(eeg_idx);
  ch_info.Name   = eeg_info.ChannelInfo.Name  (eeg_idx);
  ch_info.ID     = eeg_info.ChannelInfo.ID    (eeg_idx);
  ch_info.Type   = eeg_info.ChannelInfo.Type  (eeg_idx);
  
  eeg_info.ChannelInfo = ch_info;
end

if isfield(eeg_info, 'ActiveChannel')
  active_channel = eeg_info.ActiveChannel(eeg_idx);
  
  eeg_info.ActiveChannel = active_channel;
end

% --- EXTRA channels
if isfield(eeg_info, 'ExtraChannelInfo')
  [ex_names, ex_idx] = vb_util_omit_list(old_ch_name, new_ch_name);
  
  if ~isempty(ex_names)
    ex_names = vb_util_arrange_list(ex_names, 0);

    n_ex_channel = length(ex_names);
    ex_info.Channel_active = ones(n_ex_channel,1);
    ex_info.Channel_name   = ex_names;
    ex_info.Channel_type   = ones(n_ex_channel,1);
    ex_info.Channel_id     = old_ch_id(ex_idx);
  
    % --- set to new eeg_info
    eeg_info.ExtraChannelInfo = ex_info;
  else
    fprintf('(%s) It may not be necessary to update ExtraChannelInfo\n', ...
      mfilename);
  end
end


% --- coordinates
eeg_info = ...
  vb_eeginfo_set_sensor_position(eeg_info, pos_info.SensorPosition(pos_idx,:));

% --- spherical
[center, radius, c_type] = vb_posfile_get_sphericalinfo(pos_file);
eeg_info.Vcenter   = center;
eeg_info.Vradius   = radius;
eeg_info.CoordType = c_type;

% --- MRI_ID, TransInfo
eeg_info.MRI_ID = vb_posfile_load_mrikey(pos_file);
eeg_info = vb_info_set_transinfo(eeg_info, vb_posfile_load_transinfo(pos_file));

% --- Nchannel
eeg_info.Nchannel = n_channel;

cur_eeg.EEGinfo = eeg_info;

vb_save_struct(new_file, cur_eeg);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [pos_file, eeg_file, new_file, result] = ...
  inner_check_arguments(pos_file, eeg_file, new_file)

func_ = mfilename;
result = 0;

if isempty(pos_file)
  fprintf('(%s) pos_file is a required parameter\n', func_);
  result = 1;
  return;
end

if exist(pos_file, 'file') ~= 2
  fprintf('(%s) cannot find pos_file : %s\n', func_, pos_file);
  result = 1;
  return;
end

if isempty(eeg_file)
  fprintf('(%s) eeg_file is a required parameter\n', func_);
  result = 2;
  return;
end

if exist(eeg_file, 'file') ~= 2
  fprintf('(%s) cannot find eeg_file : %s\n', func_, eeg_file);
  result = 2;
  return;
end

if isempty(new_file)
  new_file = eeg_file;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
