function dist = vb_util_distance_of_points(points, mode, center)
% return distance of point set
% [usage]
%   dist = vb_util_distance_of_points(points, mode, center)
% [input]
%   points : <required> [NP x 3] 3-D point set (e.g. Vhead)
%     mode : <optional> calculation mode (1 or 2 or 3) [1]
%          :  =1 : <default> calculate average
%          :  =2 : minimum value of all distances
%          :  =3 : maximum value of all distances
%          :  =other : <warning> calculate average
%   center : <optional> coordinates of center point [0 0 0]
% [output]
%     dist : calculated distance
% [note]
% 
% [history]
%   2007-06-14 (Sako) rewrite
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('points', 'var') points = []; end
if ~exist('mode', 'var') mode = []; end
if ~exist('center', 'var') center = []; end
[points, mode, center] = inner_check_arguments(points, mode, center);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
points = [points(:,1)-center(1), points(:,2)-center(2), points(:,3)-center(3)];

switch mode
  case  1
    dist = ...
      sum(sqrt(points(:,1).^2 + points(:,2).^2 + points(:,3).^2))/length(points);
    
  case  2
    dist = ...
      min(sqrt(points(:,1).^2 + points(:,2).^2 + points(:,3).^2));
    
  case  3
    dist = ...
      max(sqrt(points(:,1).^2 + points(:,2).^2 + points(:,3).^2));
    
  otherwise
    warning('unexpected mode : %d, calculate average for now', mode);
    dist = ...
      sum(sqrt(points(:,1).^2 + points(:,2).^2 + points(:,3).^2))/length(points);
end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [points, mode, center] = inner_check_arguments(points, mode, center)

func_ = 'vb_util_distance_of_points';

if isempty(points)
  error('(%s)points is a required parameter', func_);
end

if isempty(mode)
  mode = 1;
end

if isempty(center)
  center = [0 0 0];
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
