function brain_parm = vb_freesurfer_get_files(brain_parm, fs_sbj_dir)
% Get freesurfer files for brain_parm/gbrain_parm
% Find files by looking at fs_subj_dir/surf [1st search]
%                                     /bem  [2nd search]
% [Usage]
%    vb_freesurfer_get_files(brain_parm, fs_sbj_dir);
%
% [Input]
%    brain_parm : brain_parm.FS_left_file
%                           .FS_right_file
%                           .FS_left_infl_file
%                           .FS_right_infl_file
%                           .FS_left_curv_file
%                           .FS_right_curv_file
%                           .FS_left_label_file
%                           .FS_right_label_file
%                           are set.
%    fs_sbj_dir : freesurfer subject directory.
%                 e.g. '/home/user/FS_SUBJECTS_DIR/SBJ1'
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin ~= 2
    error('Please check function usage');
end

%
% --- Main Procedure
%

% Check whether the specified directory is Freesurfer's one or not
fs_label_dir = fullfile(fs_sbj_dir, 'label');
[tmp, num_found_files] = search(brain_parm, fs_label_dir);
if num_found_files ~= 2
    error('This is not Freesurfer subject directory.');
end

%
% --- Find surface files
%

% candidate directories for surface data.
fs_surf_dir  = fullfile(fs_sbj_dir, 'surf');
fs_bem_dir   = fullfile(fs_sbj_dir, 'bem');
[tmp, num_found_files] = search(brain_parm, fs_surf_dir);
if num_found_files >= 6
    brain_parm = tmp;
else
    [tmp, num_found_files] = search(brain_parm, fs_bem_dir);
    if num_found_files >= 6
        brain_parm = tmp;
    else
        warning('Freesurfer files not found.');
    end
end

%
% --- Find label files
%
[brain_parm] = search(brain_parm, fs_label_dir);


function [brain_parm, num_found_files] = search(brain_parm, target_dir)
% Search target_files inside target_dir and set brain_parm fields
num_found_files = 0;
target_files = {'lh.smoothwm.asc';
                'rh.smoothwm.asc';
                'lh.inflated.asc';
                'rh.inflated.asc';
                'lh.curv.asc';
                'rh.curv.asc';
                'lh.cortex.label';
                'rh.cortex.label';
                'lh.sphere.reg.asc';
                'rh.sphere.reg.asc'};

for k=1:length(target_files)
    target_file = fullfile(target_dir, target_files{k});
    if exist(target_file, 'file') ~= 2
        continue;
    end
    switch(target_files{k})
        case 'lh.smoothwm.asc'
            brain_parm.FS_left_file = target_file;
        case 'rh.smoothwm.asc'
            brain_parm.FS_right_file = target_file;
        case 'lh.inflated.asc'
            brain_parm.FS_left_infl_file = target_file;
        case 'rh.inflated.asc'
            brain_parm.FS_right_infl_file = target_file;
        case 'lh.curv.asc';
            brain_parm.FS_left_curv_file = target_file;
        case 'rh.curv.asc';
            brain_parm.FS_right_curv_file = target_file;
        case 'lh.cortex.label'
            brain_parm.FS_left_label_file = target_file;
        case 'rh.cortex.label'
            brain_parm.FS_right_label_file = target_file;
        case 'lh.sphere.reg.asc'
            brain_parm.FS_left_sphere_file = target_file;
        case 'rh.sphere.reg.asc'
            brain_parm.FS_right_sphere_file = target_file;
        otherwise
            error('Unknown target file is found.');
    end
    num_found_files = num_found_files + 1;
end
