function [ avw, machine ] = avw_img_read(fileprefix,IMGorient,machine)
% avw_img_read - read Analyze format data image (*.img)
% 
% [ avw, machine ] = avw_img_read(filename)
% 
% filename - filename of Analyze file
% 
% (Modified by M. Sato)
% Do not change orientation of image data written in the file
% 
% Returned values:
% 
% avw.hdr - a struct with image data parameters.
% avw.img - a 3D matrix of image data (double precision).
% 
% The 3D matrix of image is written in .img file as
%  for z=1:NZ
%     for y=1:NY
%        for x=1:NX
%           read(img(x,y,z))
%        end
%     end
%  end
% 
% Default ANALYZE coordinate system is Left-handed LAS
% 
% X-Y plane is Transverse
% X-Z plane is Coronal
% Y-Z plane is Sagittal
% 
% X axis runs from patient right (low X) to patient Left (high X)
% Y axis runs from posterior (low Y) to Anterior (high Y)
% Z axis runs from inferior (low Z) to Superior (high Z)
% 
% See also: avw_hdr_read (called by this function), 
%           avw_view, avw_write, avw_img_write, avw_flip
% 


% $Revision: 1332 $ $Date:: 2011-02-24 15:57:20 +0900#$

% Licence:  GNU GPL, no express or implied warranties
% History:  05/2002, Darren.Weber@flinders.edu.au
%                    The Analyze format is copyright 
%                    (c) Copyright, 1986-1995
%                    Biomedical Imaging Resource, Mayo Foundation
%           01/2003, Darren.Weber@flinders.edu.au
%                    - adapted for matlab v5
%                    - revised all orientation information and handling 
%                      after seeking further advice from AnalyzeDirect.com
%           03/2003, Darren.Weber@flinders.edu.au
%                    - adapted for -ve pixdim values (non standard Analyze)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if ~exist('fileprefix','var'),
  msg = sprintf('...no input fileprefix - see help avw_img_read\n\n');
  error(msg);
end
if ~exist('IMGorient','var'), IMGorient = ''; end
if ~exist('machine','var'), machine = 'ieee-le'; end

if findstr('.hdr',fileprefix),
  fileprefix = strrep(fileprefix,'.hdr','');
end
if findstr('.img',fileprefix),
  fileprefix = strrep(fileprefix,'.img','');
end

% MAIN

% Read the file header
[ avw, machine ] = avw_hdr_read(fileprefix,machine);

avw = read_image(avw,IMGorient,machine);

return



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ avw ] = read_image(avw,IMGorient,machine)

fid = fopen(sprintf('%s.img',avw.fileprefix),'r',machine);
if fid < 0,
  msg = sprintf('...cannot open file %s.img\n\n',avw.fileprefix);
  error(msg);
end

% Modified by M. Sato
%ver = '[$Revision: 1332 $]';
%fprintf('\nAVW_IMG_READ [v%s]\n',ver(12:16));  tic;

% short int bitpix;    /* Number of bits per pixel; 1, 8, 16, 32, or 64. */ 
% short int datatype      /* Datatype for this image set */ 
% /*Acceptable values for datatype are*/ 
% _NONE             0
% _UNKNOWN          0  /*Unknown data type*/ 
% _BINARY           1  /*Binary             ( 1 bit per voxel)*/ 
% _UNSIGNED_CHAR    2  /*Unsigned character ( 8 bits per voxel)*/ 
% _SIGNED_SHORT     4  /*Signed short       (16 bits per voxel)*/ 
% _SIGNED_INT       8  /*Signed integer     (32 bits per voxel)*/ 
% _FLOAT           16  /*Floating point     (32 bits per voxel)*/ 
% _COMPLEX         32  /*Complex (64 bits per voxel; 2 float)/* 
% _DOUBLE          64  /*Double precision   (64 bits per voxel)*/ 
% _RGB            128  /*A Red-Green-Blue datatype*/
% _ALL            255  /*Undocumented*/

switch double(avw.hdr.dime.bitpix),
  case  1,   precision = 'bit1';
  case  8,   precision = 'uchar';
  case 16,   precision = 'int16';
  case 32,
    if     isequal(avw.hdr.dime.datatype, 8), precision = 'int32';
    else                                      precision = 'single';
    end
  case 64,   precision = 'double';
  otherwise,
    precision = 'uchar';
    fprintf('...precision undefined in header, using ''uchar''\n');
end

% read the whole .img file into matlab (faster)
fprintf('...reading %s Analyze %s image format.\n',machine,precision);

fseek(fid,0,'bof');

% adjust for matlab version
ver = version;
ver = str2num(ver(1));
if ver < 6,
  tmp = fread(fid,inf,sprintf('%s',precision));
else,
  tmp = fread(fid,inf,sprintf('%s=>double',precision));
end
fclose(fid);

% Update the global min and max values
avw.hdr.dime.glmax = max(double(tmp));
avw.hdr.dime.glmin = min(double(tmp));


%---------------------------------------------------------------
% Now partition the img data into xyz
% dim[0]      Number of dimensions in database; usually 4. 
% dim[1]      Image X dimension;  number of pixels in an image row. 
% dim[2]      Image Y dimension;  number of pixel rows in slice. 
% dim[3]      Volume Z dimension; number of slices in a volume. 
% dim[4]      Time points; number of volumes in database.

PixelDim = double(avw.hdr.dime.dim(2));
RowDim   = double(avw.hdr.dime.dim(3));
SliceDim = double(avw.hdr.dime.dim(4));

PixelSz  = double(avw.hdr.dime.pixdim(2));
RowSz    = double(avw.hdr.dime.pixdim(3));
SliceSz  = double(avw.hdr.dime.pixdim(4));

% ---- NON STANDARD ANALYZE...

% Some Analyze files have been found to set -ve pixdim values, eg
% the MNI template avg152T1_brain in the FSL etc/standard folder,
% perhaps to indicate flipped orientation?  If so, this code below
% will NOT handle the flip correctly!
if PixelSz < 0,
  warning('X pixdim < 0 !!! resetting to abs(avw.hdr.dime.pixdim(2))');
  PixelSz = abs(PixelSz);
  avw.hdr.dime.pixdim(2) = single(PixelSz);
end
if RowSz < 0,
  warning('Y pixdim < 0 !!! resetting to abs(avw.hdr.dime.pixdim(3))');
  RowSz = abs(RowSz);
  avw.hdr.dime.pixdim(3) = single(RowSz);
end
if SliceSz < 0,
  warning('Z pixdim < 0 !!! resetting to abs(avw.hdr.dime.pixdim(4))');
  SliceSz = abs(SliceSz);
  avw.hdr.dime.pixdim(4) = single(SliceSz);
end

% ---- END OF NON STANDARD ANALYZE

% --- Do not change orientation of image data
%
    
% --- Modified by M. Sato

avw.img = reshape(tmp, [PixelDim, RowDim, SliceDim]);
    
%    n = 1;
%    x = 1:PixelDim;
%    for z = 1:SliceDim,
%      for y = 1:RowDim,
%        % load Y row of X values into Z slice avw.img
%        avw.img(x,y,z) = tmp(n:n+(PixelDim-1));
%        n = n + PixelDim;
%      end
%    end

% no need to rearrange avw.hdr.dime.dim or avw.hdr.dime.pixdim
    
%
%t=toc; fprintf('...done (%5.2f sec).\n\n',t);

return

% From ANALYZE documentation...
% 
% The ANALYZE coordinate system has an origin in the lower left 
% corner. That is, with the subject lying supine, the coordinate 
% origin is on the right side of the body (x), at the back (y), 
% and at the feet (z). This means that:
% 
% +X increases from right (R) to left (L)
% +Y increases from the back (posterior,P) to the front (anterior, A)
% +Z increases from the feet (inferior,I) to the head (superior, S)
% 
% The LAS orientation is the radiological convention, where patient 
% left is on the image right.  The alternative neurological
% convention is RAS (also Talairach convention).
% 
% A major advantage of the Analzye origin convention is that the 
% coordinate origin of each orthogonal orientation (transverse, 
% coronal, and sagittal) lies in the lower left corner of the 
% slice as it is displayed.
% 
% Orthogonal slices are numbered from one to the number of slices
% in that orientation. For example, a volume (x, y, z) dimensioned 
% 128, 256, 48 has: 
% 
%   128 sagittal   slices numbered 1 through 128 (X)
%   256 coronal    slices numbered 1 through 256 (Y)
%    48 transverse slices numbered 1 through  48 (Z)
% 
% Pixel coordinates are made with reference to the slice numbers from 
% which the pixels come. Thus, the first pixel in the volume is 
% referenced p(1,1,1) and not at p(0,0,0).
% 
% Transverse slices are in the XY plane (also known as axial slices).
% Sagittal slices are in the ZY plane. 
% Coronal slices are in the ZX plane. 
% 
%----------------------------------------------------------------------------
