#!/bin/tcsh
#  csh script for running FreeSurfer to get all files for VBMEG

if ( $#argv < 3 ) then
    echo "usage: RunFreeSurfer.csh SUBJ_DIR SUBJ_ID IMAGE_FILE FREESURFER_HOME"
    echo "example: RunFreeSurfer.csh /home/cbi/rhayashi/FS_SUBJECTS_DIR SUBJ1 /home/cbi/rhayashi/MRI/rh/m3D.nii /home/cbi-data20/common/software/external/freesurfer/freesurferv4.5.0"
    exit 0
endif

### Set FreeSurfer install directory
if ( $#argv == 4) then
    setenv FREESURFER_HOME $4
else
    setenv FREESURFER_HOME /home/cbi-data20/common/software/external/freesurfer/freesurferv4.5.0
endif
echo $FREESURFER_HOME
##################### Usage #####################################
#             Run following command
#  
#   RunFreeSurfer.csh SUBJ_DIR SUBJ_ID IMAGE_FILE FREESURFER_HOME
#
# - Input argument 
#  SUBJ_DIR        : Root Dir for FS data
#  SUBJ_ID         : Subject ID , it becomes Sub-directory name
#  IMAGE_FILE      : Input MRI T1 Image file name.
#                     - NIfTI file(.nii)
#                     - LAS Analyze file(.img)
#  FREESURFER_HOME : Freesurfer install directory(optional)
# - This script execute the following process
#  1. Make cortical surface for brain model
#  2. Make registration on standard brain model in sphere
#          by matching main sulcus/gyrus structure
#     This information can be used for group analysis
#  3. Make atlas label for specifying ROI according to sulcus structure
#  4. Make inner/outer skull surface for BEM
#
# - Output files are saved in SUBJ_DIR/SUBJ_ID/bem
#  lh.* : left  hemisphere
#  rh.* : right hemisphere
#
#  - brain(cortex) model file
#  lh.smoothwm.asc : smooth cortical surface
#  rh.smoothwm.asc
#  lh.inflated.asc : inflated surface
#  rh.inflated.asc
#  lh.curv.asc     : curvature info file
#  rh.curv.asc
#  lh.white.asc    : cortical surface
#  rh.white.asc
#
#  - spherical registraation surface for group analysis
#  lh.sphere.reg.asc 
#  rh.sphere.reg.asc
#  
#  - skull surface for BEM
#  brain_surface.asc       : outer brain surface
#  inner_skull_surface.asc : inner skull surface
#  outer_skull_surface.asc : outer skull surface
#  outer_skin_surface.asc  : skin surface
#  
#  - atlas label for ROI specification
#  lh.aparc.annot 
#  rh.aparc.annot 
#  lh.aparc.a2009s.annot 
#  rh.aparc.a2009s.annot 
#  
#  - processed MRI image
#  T1.img        : normalized T1 image (Nifti)
#  brain_bem.img : skull stripped brain image  (Nifti)
#############################################################

##################### Set Env #############################

### Set Root Dir for FS data
setenv SUBJECTS_DIR $1

### Set Sub-directory name for subject
setenv SUBJECTS_ID $2

### Set Input Image file name
setenv IMAGE_FILE $3
#############################################################

### No need to modify the following

### Set Path for FreeSurfer
source $FREESURFER_HOME/SetUpFreeSurfer.csh

#############################################################
#### Make cortical surface model
# 1. Creates directory $SUBJECTS_ID
# 2. Converts input volumes to MGZ format in $SUBJECTS_ID/mri/orig, 
# 3. Runs all cortical surface reconstruction processing steps. 
#############################################################

cd $SUBJECTS_DIR

recon-all -subject $SUBJECTS_ID -i $IMAGE_FILE -all

#### Convert binary files to ascii files.

setenv BEM_DIR $SUBJECTS_DIR/$SUBJECTS_ID/bem
echo $BEM_DIR
if (! -e $BEM_DIR) then
    mkdir $BEM_DIR
endif
cd $SUBJECTS_DIR/$SUBJECTS_ID/surf

mris_convert lh.smoothwm ../bem/lh.smoothwm.asc 
mris_convert rh.smoothwm ../bem/rh.smoothwm.asc
mris_convert lh.inflated ../bem/lh.inflated.asc 
mris_convert rh.inflated ../bem/rh.inflated.asc
mris_convert -c curv lh.smoothwm ../bem/lh.curv.asc
mris_convert -c curv rh.smoothwm ../bem/rh.curv.asc
mris_convert lh.white ../bem/lh.white.asc
mris_convert rh.white ../bem/rh.white.asc


mris_convert lh.sphere.reg ../bem/lh.sphere.reg.asc
mris_convert rh.sphere.reg ../bem/rh.sphere.reg.asc

## for pial surface
mris_convert lh.pial ../bem/lh.pial.asc
mris_convert rh.pial ../bem/rh.pial.asc

#### Convert normalized T1 image to nifti
cd $SUBJECTS_DIR/$SUBJECTS_ID/mri
mri_convert --out_type nifti1 T1.mgz ../bem/T1

#### Copy label files
cd $SUBJECTS_DIR/$SUBJECTS_ID/label

cp lh.aparc.annot ../bem
cp lh.aparc.a2009s.annot ../bem
cp rh.aparc.annot ../bem
cp rh.aparc.a2009s.annot ../bem

#############################################################
#### Make skull surface model
#############################################################

cd $SUBJECTS_DIR/$SUBJECTS_ID/surf
mri_watershed -surf bem_surf -useSRAS ../mri/T1.mgz ../mri/brain_bem.mgz

#### Convert binary files to ascii files.
mris_convert lh.bem_surf_brain_surface        ../bem/brain_surface.asc
mris_convert lh.bem_surf_inner_skull_surface  ../bem/inner_skull_surface.asc
mris_convert lh.bem_surf_outer_skull_surface  ../bem/outer_skull_surface.asc
mris_convert lh.bem_surf_outer_skin_surface   ../bem/outer_skin_surface.asc

#### Convert skull stripped brain image to nifti
cd $SUBJECTS_DIR/$SUBJECTS_ID/mri
mri_convert --out_type nifti1 brain_bem.mgz ../bem/brain_bem
