function	Vpath = plane_cross_section(V,F,V1,V2,xx,dmax)
% Find the shortest path from V1 to V2 along surface
% The plane is defined by (V2 - V1) & the normal direction 'xn'
% The path is cross section of surface and plane
% 
%   Vpath = plane_cross_section(V,F,V1,V2,xx,dmax)
% 
% --- Input
% V  : voxcel coordinate of the surface [Nvertex x 3]
% F  : patch index of the surface       [Npatch  x 3]
% V1 : start point on the surface
% V2 : end point on the surface
% xn :  mean normal direction at V1 & V2
% --- Output
% Vpath : set of point coordinates along shotrest path (Npath x 3)
% dmax  : search distance from cut plane (= 0.05 m)
%         maximum edge length of surface patch
% --- Intersection point at Z = 0
% Line : V1 = [X1 Y1 Z1] to V2 = [X2 Y2 Z2]
%      : V  = (V1 - V2) * t + V2 
% z = 0 : 
% x = (X1*Z2 - X2*Z1)/(Z2 - Z1)
% y = (Y1*Z2 - Y2*Z1)/(Z2 - Z1)
%
% 2015-5-13 M. Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if ~exist('dmax','var'),  dmax  = 0.05; end;

% Make local XYZ coordinate (origine = V1)
Xdir = V2 -V1;
Xdir = Xdir/sqrt(sum((Xdir.^2)));
Zdir = vb_cross2(Xdir,xx);
Zdir = Zdir/sqrt(sum((Zdir.^2)));
Ydir = vb_cross2(Zdir,Xdir);
Ydir = Ydir/sqrt(sum((Ydir.^2)));

% X coordinates of start & end points
x1 = 0;
x2 = (V2 -V1)*Xdir(:);

% rotation matrix
Rot = [Xdir(:), Ydir(:), Zdir(:)];

% Coordinate transform
% XYZ = (V - V1)*Rot

XYZ = vb_repadd(V, - V1)*Rot;
X = XYZ(:,1);
Y = XYZ(:,2);
Z = XYZ(:,3);

%
% ---- Find set of intersection points at Z = 0
%

% The plane is defined by Z = 0
% Start & End points are [0, 0, 0] & [x2, 0, 0]

% Flag indicate upper & lower side of plane within dmax
Uix = (Z >= 0 & Z <= dmax);	% flag of upper side of plane
Lix = (Z <  0 & Z >= -dmax);% flag of lower side of plane

% Flag for patch index
FUix = Uix(F); 
FLix = Lix(F); 

% Patch include both upper and lower side of plane within dmax
Flist = find(sum(FUix,2) > 0 & sum(FLix,2) > 0);
Nlist = length(Flist);

x  = [];	% x-coordinate to intersect the plane Z = 0
y  = [];	% y-coordinate to intersect the plane Z = 0

% Loop for patch include both upper and lower side of plane
for j=1:Nlist
	n  = Flist(j);	% patch id
	id = F(n,:);	% vertex index of patch
	up = FUix(n,:);	% Upper side flag of each vertex in the n-th patch
	dn = FLix(n,:);	% Lower side flag of each vertex in the n-th patch
	
	% Check two vertex of edge are in opposite side
	% : Check edge intersect the plane Z = 0
	%   and add intersection points to the list
	if (up(1)*dn(2) + up(2)*dn(1)) > 0,
		% Two vertex points of edge
		X1 = X(id(1));	X2 = X(id(2));
		Y1 = Y(id(1));	Y2 = Y(id(2));
		Z1 = Z(id(1));	Z2 = Z(id(2));
		
		if Z1 == Z2,
			% line is on the Z = 0 plane
			x = [x; X1; X2];
			y = [y; Y1; Y2];
		else
			% Intersection point at Z = 0
			x = [x; (X1*Z2 - X2*Z1)/(Z2 - Z1)];
			y = [y; (Y1*Z2 - Y2*Z1)/(Z2 - Z1)];
		end
	end
	
	if (up(2)*dn(3) + up(3)*dn(2)) > 0,
		% Two vertex points of edge
		X1 = X(id(2));	X2 = X(id(3));
		Y1 = Y(id(2));	Y2 = Y(id(3));
		Z1 = Z(id(2));	Z2 = Z(id(3));
		
		if Z1 == Z2,
			% line is on the Z = 0 plane
			x = [x; X1; X2];
			y = [y; Y1; Y2];
		else
			% Intersection point at Z = 0
			x = [x; (X1*Z2 - X2*Z1)/(Z2 - Z1)];
			y = [y; (Y1*Z2 - Y2*Z1)/(Z2 - Z1)];
		end
	end
	
	if (up(3)*dn(1) + up(1)*dn(3)) > 0,
		% Two vertex points of edge
		X1 = X(id(3));	X2 = X(id(1));
		Y1 = Y(id(3));	Y2 = Y(id(1));
		Z1 = Z(id(3));	Z2 = Z(id(1));
		
		if Z1 == Z2,
			% line is on the Z = 0 plane
			x = [x; X1; X2];
			y = [y; Y1; Y2];
		else
			% Intersection point at Z = 0
			x = [x; (X1*Z2 - X2*Z1)/(Z2 - Z1)];
			y = [y; (Y1*Z2 - Y2*Z1)/(Z2 - Z1)];
		end
	end
end

NP = length(x);

%
% ---- Search path from V1 to V2
%

% start point = V1 = [0 0 0]
xp = 0;
yp = 0;

xpath = xp;
ypath = yp;
dpath = [];

% extract self point
dx = (x - xp).^2 + (y - yp).^2;
[dmin, j0] = min(dx); % dmin = 0

% remove self point from list
x(j0(1)) = [];
y(j0(1)) = [];
	
% Find path from [0 0] to [x2 0]
for n = 1:NP
	% select direction
	ix = find(x > xp);
	% find nearest point
	dx = (x(ix) - xp).^2 + (y(ix) - yp).^2;
	[dmin, j0] = min(dx);
	j0 = ix(j0(1));
	
	xp = x(j0);
	yp = y(j0);
	
	x(j0) = [];
	y(j0) = [];
	
	% add nearest point
	xpath = [xpath; xp];
	ypath = [ypath; yp];
	dpath = [dpath; sqrt(dmin)];
	
	if abs(xp-x2)<eps & abs(yp-0)<eps,	break; end;
end

% Back Coordinate transform
% XYZ = (V - V1)*Rot
%  V  = XYZ*R' + V1

Npath = length(xpath);
xyz   = [xpath ypath zeros(Npath,1)];
Vpath = xyz*Rot' ;
Vpath = vb_repadd(Vpath, V1);
