function	[V,F,X,Info] = make_cylinder(X,dN,dr)
% Make cylinder around set of points which represent sequential line
%  [V,F,X] = make_cylinder(X,dN,dr)
% X  : set of points 3D coordinates [N x 3]
% dN : number of points consist cylinder
%      = 6 means hexagonal cylinder will be made
% dr : radius of cylinder (same unit with X)
% V  : vertex coordinate for cylindrical lines
% F  : patch index for cylindrical lines
%
% 2014-11-10 Masa-aki Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%ʬ˱ä dN ꡢpatch

[N, d] = size(X);
if d ~= 3, error('data dimension error'); end
if N < 2, error('There is only one point'); end

% dx : line segment direction [N x 3]
dx = diff(X);
dx = [dx; dx(end,:)];

% Find reverse direction
dxs = sum(dx(1:N-1,:).*dx(2:N,:),2);
dxs = [1; dxs];
irv = find(dxs < 0);
Nrev = length(irv);

% Insert duplicate points at reverse points
Xnew = [];
ist = 1;
for m = 1:Nrev
	n = irv(m);
	Xnew = [Xnew; X(ist:n,:); X(n,:)];
	ist = n+1;
end

if ist <= N
	Xnew = [Xnew; X(ist:N,:)];
end

if size(Xnew,1) ~= (N+Nrev), error('Insert error'); end;

X = Xnew;
N = size(X,1);

% Recalculate line segment direction [N x 3]
dx = diff(X);
dx = [dx; dx(end,:)];

% Reverse direction check
dxs = sum(dx(1:N-1,:).*dx(2:N,:),2);
dxs = [1; dxs];
irv = find(dxs < 0);
Nrev = length(irv);

[dY,dZ] = make_orthogonal_vec(dx);

% Find zero difference vector
dxx = sum(dx.^2,2);
zx  = find(dxx == 0);
Nzero = length(zx);

Info.Nzero = Nzero;
Info.Nrev  = Nrev;

% X(n-1) -> X(n) = X(n+1) -> X(n+2)
%    dx(n-1)  dx(n)=0  dx(n+1)
% X(n-1) -> X(n)  : stop line
%             dx(n)=dx(n-1)
%      start line : X(n+1) -> X(n+2)

% Copy preceding direction to stop connected cylinder
for m=1:Nzero
	n = zx(m);
	if n > 1
		dx(n,:) = dx(n-1,:);
		dY(n,:) = dY(n-1,:);
		dZ(n,:) = dZ(n-1,:);
	end
end

% incremental angle of cylindrical points
ang = 2*pi/dN;
da  = ang*(0:dN-1);
% [Y,Z] coordinate of cylindrical points
y = cos(da);
z = sin(da);

% make dN points for each N points
V   = zeros(dN*N,3);
xyz = zeros(N,3,dN);

% make cylindrical points around X with radius dr
for m=1:dN
	xyz(:,:,m) = X + (dY*y(m) + dZ*z(m))*dr;
end

% find nearest point from xyz(n,:,:) to xyz(n+1,:,:)
% which is cylindrical points in next X(n+1,:)
% and reorder the next cylindrical points xyz(n+1,:,:)
for n=1:N-1
	% search for min distance between xyz(n,:,:) and xyz(n+1,:,:)
	xp = reshape(xyz(n,:,:),[3 dN]);
	xn = reshape(xyz(n+1,:,:),[3 dN]);
	dd = sq_distance(xp,xn);
	
	[dmin, jmin] = min(dd,[],2);
	[dtmp, kmin] = min(dmin);
	% min distance : xyz(n,:,kmin) - xyz(n+1,:,jmin(kmin))
	
	imin = jmin(kmin) - kmin + 1;
	
	% reorder the next cylindrical points xyz(n+1,:,:)
	ind = imin + (0:dN-1);
	ind = mod(ind-1,dN) + 1;
	
	xyz(n+1,:,:) = xyz(n+1,:,ind);
end

% cylindrical points of X(n,:)
% : xyz(n,:,m) 
% = V( dN*(n-1) + m, :)
nj = 0:dN:(dN*(N-1));

for m=1:dN
	V(nj + m,:) = xyz(:,:,m);
end

%--- Example of vertex & patch
%  1  2  3  1
%  |/ |/ |/ |
%  4  5  6  4
%---  n=1, dN=3
%  F = [n n+1 n+dN; 
%         n+1 n+dN n+1+dN] 
% There are 2*dN patches for pair of consecutive cylindrical points
F  = zeros(2*dN*(N-1),3);

% make patch index
for i=1:N-1
	np = dN*(i-1);  % start point index
	nf = 2*dN*(i-1);% start patch index
	
	for n=1:dN
		if n==dN, 
			m = 1;
		else
			m = n+1;
		end
		
		%  F = [n n+1 n+dN; 
		%         n+1 n+dN n+1+dN] 
		F(nf+n,:)    = [n   m    n+dN] + np;
		F(nf+dN+n,:) = [m   m+dN n+dN] + np;
	end
end
