function movie_top_view_create(fiber_tracking_file, ...
    current_file, movie_file, Timefrom, Tfreq)
% Create movie file. The view point is top side of cortex.
%    F
% L     R :  Front, Back, Left, Right
%    B  
%
% [Input]
%      fiber_tracking_file : [input file]  fiber tracking result file.
%             current_file : [input file]  MAR dynamics file.
%               movie_file : [output file] output movie file.(empty : no output)
%                 Timefrom : [optional]    Time starts from (ms).
%                    Tfreq : [optional]    time frequency(Hz) : default:1000
%
%
% [History]
%    2014-11-10 Masa-aki Sato  Created as a test_ar_cylinder.m
%    2015-12-07 rhayashi       Rename, fine-tuning and added all arguments.
%    2017-12-12 Yusuke Takeda  Modified details
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if exist(fiber_tracking_file, 'file') ~= 2
    error('fiber_tracking_file not found.');
end
if exist(current_file, 'file') ~= 2
    error('current_file not found.');
end
if ~exist('movie_file', 'var')
    movie_file = [];
elseif ~isempty(movie_file)
    [p_, f_, e_] = fileparts(movie_file);
    if exist(p_, 'dir') ~= 7
        mkdir(p_);
    end
end

if ~exist('Timefrom', 'var')
    Timefrom = 0;     % [ms]
end
if ~exist('Tfreq', 'var')
    Tfreq = 1000; % [Hz]
end

%
% --- Main Procedure
%

%------- Plot parameter
plot_para.conn_file = fiber_tracking_file;
plot_para.curr_file = current_file;

%--- figure parameter
plot_para.fig_unit = 'pixels';
plot_para.fig_pos  = [200, 200, 768, 720];

%--- brain surface
plot_para.faceclr = [0.8 0.7 0.6];
plot_para.faceclr0 = [1 1 1];% face color of brain
plot_para.alpha = [0.1 0.1 0.01];% alpha value of brain

%--- face surface
plot_para.face_on    = 1; % =1:plot face
plot_para.alpha_face = 0.5; % Initial alpha value of face
plot_para.face_file  = which('Subject.face.mat');
plot_para.zoom_factor= 1.3; % zoom factor when a face disappears

%--- area surface
plot_para.alpha_area = [0.8 0.08 0.01]; % alpha value of area
plot_para.cbox = 0;                     % show area color update dialog

%--- view angle
plot_para.angle = [0 90];% initial view angle
plot_para.dang = [0 0];% increment of view angle in one step
plot_para.Ntime = 0;% number of times for rotation

%--- ROI area surface
plot_para.roi = 0; %=1: show ROI surface

% Time length & speed
plot_para.Tfreq   = Tfreq;
plot_para.Timefrom= Timefrom;  % time (ms) starts from (for display)
plot_para.Tstay   = 100; % time length (sample) for first non-rotate state
plot_para.Tstop   = 600; % time point (sample) of stop rotation.

%--- connection line
plot_para.Nskip = 1;% skip number for connections selection
plot_para.Nstep = 1;% skip number along path (neighbor distance= Nstep*0.2mm)
%plot_para.Ndivide = 50;% number of division of connection line
%plot_para.Nperiod = 100;% time window length

%--- activity transition movie
plot_para.pmove   = 1;%=1: show activity transition movie
plot_para.pause1  = 0.01;% pause time(sec)
plot_para.pause2  = 0.05;%           (sec)

plot_para.dN    = 10;% number of cylinder points
plot_para.dr    = 0.3;% radius of cylinder [mm]
%plot_para.sigma = 0.1;% input noise sd
plot_para.omega = 0.5;% oscillation speed of dynamics [no dynamics file case]

plot_para.show_max = 1; % max value for caxis
plot_para.abs_mode = 1;%=1: show absolute value of activity
plot_para.max_norm = 1;%=1: normalize activity by max value for each vertex
plot_para.wmulti  = 0; %=1: multiply MAR weight
plot_para.vertex_show = 2;% =1: show vertex activity
plot_para.Msize   = 10; % marker size
plot_para.th_show = 0.8; % threshold to show vertex activity

%--- message box for stop rotation
plot_para.mbox = 0;
plot_para.video = 0;
if ~isempty(movie_file)
    plot_para.video = 1;
    plot_para.video_dir = p_;
end

% color map for connection line
color_parm.Nlevel  = 64; % gradation color level
color_parm.Nwidth  = 13; % active color region length
color_parm.Ncycle  = 20; % cycle length
color_parm.Nrepeat = 5;  % repeat number of active cycle
color_parm.Nback   = 21;  % repeat number of background cycle
color_parm.mode = 'gradation'; % color map mode ('gradation','jet','hot')

%----- colormap for connection
color_parm.active_color = 'white'; % color name for active state
color_parm.rest_color   = 'navyblue';  % color name for rest state
color_parm.back_color   = 'blue';      % color name for background

%---> Set colormap
%C1  = cmap_move(color_parm);
C1 = cmap_move2(color_parm);
%colorbar; colormap(C1);

%----- colormap for vertex
color_parm.active_color = 'yellow';      % color name for active state
color_parm.rest_color   = 'orange';      % color name for rest state
color_parm.back_color   = 'bottlegreen'; % color name for background
%---> Set colormap
C2 = cmap_move2(color_parm);

%----- set colormap
plot_para.cmap  = C1;
plot_para.cmap2 = C2;

%----- Load Connection data
connection = load_connection_from_vmni_connect_file(plot_para.conn_file);

%----- Create movie
close all;
[hfig, Q, id] = plot_cylinder_prepare(connection, plot_para);
set(gca, 'Position', [0.1300    0.2300    0.7750    0.8150]);
%    colorbar;
plot_cylinder_anime(plot_para, hfig, Q, id);

if plot_para.video
    [success, message] = movefile(fullfile(p_, 'cylinder.avi'), movie_file);
    if success
        disp(['top view: ' movie_file ' was created.']);
    else
        error(message);
    end
end
