function [omega] = calc_MARinput(MAR, Delta, Z)
% Compute an example of input signals to an MAR model
% (input computed by subtracting current sources with one-step prediction 
% from the estimated current sources)
%
% Input
%     MAR: MAR matrix                 [Nvact x Nvact, sparse double]
%   Delta: Time lag matrix            [Nvact x Nvact, sparse double]
%       Z: Estimated current sources  [1 x 1, cell][Nvact x Nt, double]
%
% Output
%   omega: Input signals to an MAR model [Nvact x Nt, double]
%
% 2015/10/07 M.Fukushima
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Estimated current sources
Jfilt = Z{1};

% # of sources
Nvact = size(MAR,1);
% # of time instances
Nt = size(Jfilt,2);
% Binarized structural connectivity matrix
Ind = (Delta~=0);

% Connection numbers
sum_ind = sum(Ind,2);
% Time lags
unidlt = full(setdiff(unique(Delta),0));

% Store matrix entries into a smaller size of matrix
Indx   = zeros(Nvact,max(sum_ind));
Deltax = zeros(Nvact,max(sum_ind));
MARx   = zeros(Nvact,max(sum_ind));
for nv = 1:Nvact
  Indx(nv,1:sum_ind(nv))   = find(Ind(nv,:)==1);
  Deltax(nv,1:sum_ind(nv)) = Delta(nv,Delta(nv,:)~=0);
  MARx(nv,1:sum_ind(nv))   = MAR(nv,Ind(nv,:)~=0);
end

% One step forward prediction with an MAR model
Jpred = zeros(Nvact, Nt);
for t = 1:Nt
  Jtmp = [zeros(Nvact, unidlt(end)) Jfilt];
  t_end = t + unidlt(end);
  
  % Prediction [MEX ver.]
  AJpred = atimesj(MARx, Indx, Deltax, sum_ind, ...
    Jtmp(:,t_end:-1:(t_end-unidlt(end))));
%   % Prediction [Matlab ver.]
%   AJpred = zeros(Nvact,1);
%   for nv = 1:Nvact
%     for ii = 1:sum_ind(nv)
%       AJpred(nv) = AJpred(nv) + ...
%         MARx(nv,ii) * Jtmp(Indx(nv,ii),t_end-Deltax(nv,ii));
%     end
%   end

  % Current sources of one step prediction
  Jpred(:,t) = AJpred;
end

% Input signals to an MAR model
omega = Jfilt - Jpred;
