function dmri_job_t1_preprocess(t1_parm)
% Preprocess T1 las analyze image and create files for DTI processing.
% threshold is used for creating skull stripping image.
%
% [Usage]
%    dmri_job_t1_preprocess(t1_parm);
%
% [Input]
%    t1_parm : parameter for processing.
%         .mri_file          : LAS analyze filename(.img)            [string]
%                                      or
%                              NIfTI file(.nii)
%                              This file can be created from DICOM files 
%                              using VBMEG function: convert_dicom_nifti.m
%                              
%         .threshold         : <<optional>>                    [value]
%                              fractional intensity threshold (0->1).
%                              default is 0.5; smaller values give larger brain
%                              outline estimates.
%         .output_dir        : <<optional>>                    [string]
%                                output directory. 
%                                if not specified, current directory is set.
%         .output_basename   : <<optional>>                    [string]
%                                basename of output files.
%                                if not specified, 'struct' is set.
%
% [Output]
%    none
%
% Output files:
%    output_dir/output_basename.nii.gz
%              /output_basename_brain.nii.gz
%              /output_basename_mask.nii.gz
% [Note]
%   File format - .nii.gz is a standard format for FSL utility.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin ~= 1
    error('Please check input arguments');
end
if ~isfield(t1_parm, 'mri_file')
    error('mri_file not specified.');
end
if exist(t1_parm.mri_file, 'file') ~= 2
    error('mri_file not found.');
end
mri_file = t1_parm.mri_file;

% output directory
if isfield(t1_parm, 'output_dir')
    output_dir = t1_parm.output_dir;
else
    output_dir = pwd;
end
if exist(output_dir, 'dir') ~= 7
    mkdir(output_dir);
    if exist(output_dir, 'dir') ~= 7
        error('failed to create output directory.');
    end
end
% output basename
if isfield(t1_parm, 'output_basename')
    output_basename = t1_parm.output_basename;
else
    output_basename = 'struct';
end

% threshold
if isfield(t1_parm, 'threshold')
    threshold = t1_parm.threshold;
else
    threshold = 0.5;
end

%
% --- Main Procedure
%

% Create gzipped NIfTI file - struct.nii.gz
struct_nifti_file = [output_dir, '/', output_basename '.nii'];
command  = ['mri_convert --out_type nii ' mri_file, ' ' struct_nifti_file];
dmri_system(command);
command2 = ['gzip -f ' struct_nifti_file];
system(command2);

% Create skull stripped brain file and mask file
brain_image_extract([struct_nifti_file, '.gz'], threshold);
